// LAF OS Library
// Copyright (C) 2018-2024  Igara Studio S.A.
// Copyright (C) 2017-2018  David Capello
//
// This file is released under the terms of the MIT license.
// Read LICENSE.txt for more information.

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "os/x11/window.h"

#include "base/debug.h"
#include "base/hex.h"
#include "base/split_string.h"
#include "base/string.h"
#include "base/thread.h"
#include "base/trim_string.h"
#include "gfx/border.h"
#include "gfx/rect.h"
#include "gfx/region.h"
#include "os/event.h"
#include "os/event_queue.h"
#include "os/surface.h"
#include "os/system.h"
#include "os/window_spec.h"
#include "os/x11/cursor.h"
#include "os/x11/keys.h"
#include "os/x11/screen.h"
#include "os/x11/system.h"
#include "os/x11/x11.h"
#include "os/x11/xinput.h"

#include <array>
#include <map>
#include <set>

#define KEY_TRACE(...)
#define EVENT_TRACE(...)

#define LAF_X11_DOUBLE_CLICK_TIMEOUT 250

// TODO the window name should be customized from the CMakeLists.txt
//      properties (see OS_WND_CLASS_NAME too)
#define LAF_X11_WM_CLASS "Aseprite"

const int _NET_WM_STATE_REMOVE = 0;
const int _NET_WM_STATE_ADD    = 1;

const int _NET_WM_MOVERESIZE_SIZE_TOPLEFT      = 0;
const int _NET_WM_MOVERESIZE_SIZE_TOP          = 1;
const int _NET_WM_MOVERESIZE_SIZE_TOPRIGHT     = 2;
const int _NET_WM_MOVERESIZE_SIZE_RIGHT        = 3;
const int _NET_WM_MOVERESIZE_SIZE_BOTTOMRIGHT  = 4;
const int _NET_WM_MOVERESIZE_SIZE_BOTTOM       = 5;
const int _NET_WM_MOVERESIZE_SIZE_BOTTOMLEFT   = 6;
const int _NET_WM_MOVERESIZE_SIZE_LEFT         = 7;
const int _NET_WM_MOVERESIZE_MOVE              = 8;
const int _NET_WM_MOVERESIZE_SIZE_KEYBOARD     = 9;
const int _NET_WM_MOVERESIZE_MOVE_KEYBOARD    = 10;
const int _NET_WM_MOVERESIZE_CANCEL           = 11;

namespace os {

namespace {

// This is the expected data for the _MOTIF_WM_HINTS property. Each
// field must be long/unsigned long as they are passed as 5
// XA_CARDINAL values through XChangeProperty() function.
struct MotifHints {
  enum class Flags : long {
    kNone = 0,
    kDecorations = 2,
  };
  enum class Decorations : long {
    kNone = 0,
    kAll = 1,
  };
  Flags flags = Flags::kNone;
  long functions = 0;
  Decorations decorations = Decorations::kNone;
  long inputMode = 0;
  long status = 0;
};

// Event generated by the window manager when the close button on the
// window is pressed by the userh.
Atom WM_PROTOCOLS = 0;
Atom WM_DELETE_WINDOW = 0;
Atom _NET_FRAME_EXTENTS = 0;
Atom _NET_REQUEST_FRAME_EXTENTS = 0;
Atom _NET_WM_STATE = 0;
Atom _NET_WM_STATE_MAXIMIZED_VERT;
Atom _NET_WM_STATE_MAXIMIZED_HORZ;
Atom _NET_WM_ALLOWED_ACTIONS = 0;

// Atoms used for the XDND protocol
Atom XdndAware = 0;
Atom XdndPosition = 0;
Atom XdndStatus = 0;
Atom XdndActionCopy = 0;
Atom XdndDrop = 0;
Atom XdndFinished = 0;
Atom XdndSelection = 0;
Atom URI_LIST = 0;
::Window g_dndSource = 0;
gfx::Point g_dndPosition;

// See https://bugs.freedesktop.org/show_bug.cgi?id=12871 for more
// information, it looks like the official way to convert a X Window
// into our own user data pointer (WindowX11 instance) is using a map.
std::map<::Window, WindowX11*> g_activeWindows;

// Last time an XInput event was received, it's used to avoid
// processing mouse motion events that are generated at the same time
// for the XInput devices.
Time g_lastXInputEventTime = 0;

// Set of all currently pressed keys - used during processing KeyPress or KeyRelease
// event
std::set<KeySym> g_pressedKeys;

bool is_mouse_wheel_button(int button)
{
  return (button == Button4 || button == Button5 ||
          button == 6 || button == 7);
}

gfx::Point get_mouse_wheel_delta(int button)
{
  gfx::Point delta(0, 0);
  switch (button) {
    // Vertical wheel
    case Button4: delta.y = -1; break;
    case Button5: delta.y = +1; break;
    // Horizontal wheel
    case 6: delta.x = -1; break;
    case 7: delta.x = +1; break;
  }
  return delta;
}

std::string decode_url(const std::string& in)
{
  std::string out;
  out.reserve(in.size());

  int i;
  if (std::strncmp(in.c_str(), "file://", 7) == 0)
    i = 7;
  else
    i = 0;

  for (; i<in.size(); ++i) {
    auto c = in[i];
    if (c == '%' && i+2 < in.size()) {
      c = ((base::hex_to_int(in[i+1]) << 4) |
           (base::hex_to_int(in[i+2])));
      i += 2;
    }
    out.push_back(c);
  }

  base::trim_string(out, out);
  return out;
}

} // anonymous namespace

// static
bool WindowX11::g_translateDeadKeys = false;

// static
WindowX11* WindowX11::getPointerFromHandle(::Window handle)
{
  auto it = g_activeWindows.find(handle);
  if (it != g_activeWindows.end())
    return it->second;
  return nullptr;
}

// static
size_t WindowX11::countActiveWindows()
{
  return g_activeWindows.size();
}

// static
void WindowX11::addWindow(WindowX11* window)
{
  ASSERT(g_activeWindows.find(window->x11window()) == g_activeWindows.end());
  g_activeWindows[window->x11window()] = window;
}

// static
void WindowX11::removeWindow(WindowX11* window)
{
  auto it = g_activeWindows.find(window->x11window());
  ASSERT(it != g_activeWindows.end());
  if (it != g_activeWindows.end()) {
    ASSERT(it->second == window);
    g_activeWindows.erase(it);
  }
}

WindowX11::WindowX11(::Display* display, const WindowSpec& spec)
  : m_display(display)
  , m_gc(nullptr)
  , m_xic(nullptr)
  , m_scale(spec.scale())
  , m_lastMousePos(-1, -1)
  , m_lastClientSize(0, 0)
  , m_doubleClickButton(Event::NoneButton)
  , m_borderless(spec.borderless())
  , m_closable(spec.closable())
  , m_maximizable(spec.maximizable())
  , m_minimizable(spec.minimizable())
  , m_resizable(spec.resizable())
  , m_transparent(spec.transparent())
{
  // Cache some atoms (TODO improve this to cache more atoms)
  if (!_NET_FRAME_EXTENTS) {
    _NET_FRAME_EXTENTS = XInternAtom(m_display, "_NET_FRAME_EXTENTS", False);
    _NET_REQUEST_FRAME_EXTENTS = XInternAtom(m_display, "_NET_REQUEST_FRAME_EXTENTS", False);
  }
  if (!_NET_WM_STATE) {
    _NET_WM_STATE = XInternAtom(m_display, "_NET_WM_STATE", False);
    _NET_WM_STATE_MAXIMIZED_VERT = XInternAtom(m_display, "_NET_WM_STATE_MAXIMIZED_VERT", False);
    _NET_WM_STATE_MAXIMIZED_HORZ = XInternAtom(m_display, "_NET_WM_STATE_MAXIMIZED_HORZ", False);
  }
  if (!_NET_WM_ALLOWED_ACTIONS)
    _NET_WM_ALLOWED_ACTIONS = XInternAtom(m_display, "_NET_WM_ALLOWED_ACTIONS", False);

  // Initialize special messages (just the first time a WindowX11 is
  // created)
  if (!WM_PROTOCOLS)
    WM_PROTOCOLS = XInternAtom(m_display, "WM_PROTOCOLS", False);
  if (!WM_DELETE_WINDOW)
    WM_DELETE_WINDOW = XInternAtom(m_display, "WM_DELETE_WINDOW", False);

  // Get a 32 bpp visual information for transparent windows.
  XVisualInfo vi;
  if (m_transparent) {
    Status s =
      XMatchVisualInfo(m_display,
                       DefaultScreen(m_display),
                       32, TrueColor, &vi);
    if (s == 0) {
      // This X11 server doesn't support transparent windows/RGBA
      // images.
      m_transparent = false;
    }
  }

  const ::Window root = XDefaultRootWindow(m_display);

  XSetWindowAttributes swa;
  int swa_mask = CWEventMask;
  swa.event_mask = (StructureNotifyMask | ExposureMask | PropertyChangeMask |
                    EnterWindowMask | LeaveWindowMask | FocusChangeMask |
                    ButtonPressMask | ButtonReleaseMask | PointerMotionMask |
                    KeyPressMask | KeyReleaseMask);
  if (m_transparent) {
    // If one of these attributes is not specified, XCreateWindow()
    // will crash/fail with a BadMatch error.
    swa.background_pixmap = None;
    swa.border_pixel = 0;
    swa.colormap = XCreateColormap(m_display, root, vi.visual, AllocNone);
    swa_mask |= CWBackPixmap | CWBorderPixel | CWColormap;
  }

  // We cannot use the override-redirect state because it removes too
  // much behavior of the WM (cannot resize the custom frame as other
  // regular windows in the WM, etc.)
  //swa.override_redirect = (spec.borderless() ? True: False);

  gfx::Rect rc;

  if (!spec.frame().isEmpty())
    rc = spec.frame();
  else
    rc = spec.contentRect();

  m_window = XCreateWindow(
    m_display, root,
    rc.x, rc.y, rc.w, rc.h, 0,
    (m_transparent ? vi.depth: CopyFromParent),
    InputOutput,
    (m_transparent ? vi.visual: CopyFromParent),
    swa_mask, // Do not use CWOverrideRedirect
    &swa);

  if (!m_window)
    throw std::runtime_error("Cannot create X11 window");

  setWMClass(LAF_X11_WM_CLASS);

  // Special frame for this window
  if (spec.floating()) {
    // We use _NET_WM_WINDOW_TYPE_UTILITY for floating windows
    const Atom _NET_WM_WINDOW_TYPE = XInternAtom(m_display, "_NET_WM_WINDOW_TYPE", False);
    const Atom _NET_WM_WINDOW_TYPE_UTILITY = XInternAtom(m_display, "_NET_WM_WINDOW_TYPE_UTILITY", False);
    const Atom _NET_WM_WINDOW_TYPE_NORMAL = XInternAtom(m_display, "_NET_WM_WINDOW_TYPE_NORMAL", False);
    if (_NET_WM_WINDOW_TYPE &&
        _NET_WM_WINDOW_TYPE_UTILITY &&
        _NET_WM_WINDOW_TYPE_NORMAL) {
      // We've to specify the window types in order of preference (but
      // must include at least one of the basic window type atoms).
      std::vector<Atom> data = { _NET_WM_WINDOW_TYPE_UTILITY,
                                 _NET_WM_WINDOW_TYPE_NORMAL };
      XChangeProperty(
        m_display, m_window, _NET_WM_WINDOW_TYPE,
        XA_ATOM, 32, PropModeReplace,
        (const unsigned char*)data.data(), data.size());
    }
  }

  // To remove the borders and keep the window behavior of the Window
  // Manager (e.g. Super key + mouse to resize/move the window), we
  // can set the _MOTIF_WM_HINTS decorations to 0 (without
  // decorations).
  //
  // The alternatives (using _NET_WM_WINDOW_TYPE or override-redirect)
  // are useless because they remove the default behavior of the
  // operating system (making a complete "naked" window without
  // behavior at all).
  {
    MotifHints hints;
    hints.flags = MotifHints::Flags::kDecorations;
    hints.decorations = (spec.borderless() ? MotifHints::Decorations::kNone:
                                             MotifHints::Decorations::kAll);

    const Atom _MOTIF_WM_HINTS = XInternAtom(m_display, "_MOTIF_WM_HINTS", False);
    XChangeProperty(
      m_display, m_window,
      _MOTIF_WM_HINTS,
      _MOTIF_WM_HINTS,  // Instead of XA_CARDINAL here goes _MOTIF_WM_HINTS too
      32, PropModeReplace,
      (const unsigned char*)&hints,
      sizeof(hints) / sizeof(long));

    static_assert(sizeof(hints) / sizeof(long) == 5, "Invalid MotifHints struct");
  }

  // Receive stylus/eraser events
  X11::instance()->xinput()->selectExtensionEvents(m_display, m_window);

  // Change preferred origin/size for the window (this should be used by the WM)
  {
    XSizeHints* hints = XAllocSizeHints();
    hints->flags  =
      PPosition | PSize |
      PResizeInc | PWinGravity;
    hints->x = rc.x;
    hints->y = rc.y;
    hints->width  = rc.w;
    hints->height = rc.h;
    hints->width_inc = m_scale;
    hints->height_inc = m_scale;
    hints->win_gravity = SouthGravity;
    XSetWMNormalHints(m_display, m_window, hints);
    XFree(hints);
  }

  XMapWindow(m_display, m_window);

  // In case the user wants to set the initial window bounds as the
  // frame bounds, and as X11 expects the content bounds in
  // XMoveResizeWindow(), we've to remove the frame extents from the
  // specified spec.frame() rectangle. Anyway the frame extents is not
  // yet available here, so we have to send a
  // _NET_REQUEST_FRAME_EXTENTS event to request the frame extents.
  if (!spec.borderless() && !spec.frame().isEmpty()) {
    if (requestX11FrameExtents()) {
      getX11FrameExtents();
      rc.shrink(m_frameExtents);
    }
  }

  // Set the window position and size as the position is not correctly
  // used from the XCreateWindow() or XSizeHints.
  XMoveResizeWindow(m_display, m_window,
                    rc.x, rc.y, rc.w, rc.h);

  XSetWMProtocols(m_display, m_window, &WM_DELETE_WINDOW, 1);

  if (spec.floating() && spec.parent()) {
    ASSERT(static_cast<WindowX11*>(spec.parent())->m_window);
    XSetTransientForHint(
      m_display,
      m_window,
      static_cast<WindowX11*>(spec.parent())->m_window);
  }

  m_gc = XCreateGC(m_display, m_window, 0, nullptr);

  XIM xim = X11::instance()->xim();
  if (xim) {
    m_xic = XCreateIC(xim,
                      XNInputStyle, XIMPreeditNothing | XIMStatusNothing,
                      XNClientWindow, m_window,
                      XNFocusWindow, m_window,
                      nullptr);
  }

  // Adding the XdndAware property to the window we specify that we
  // support drag-and-drop operations (so we can drop files to
  // generate Event::Type::DropFiles events from this specific
  // window).
  //
  // TODO add support for other formats (e.g. dropping images?)
  if (!XdndAware) {
    XdndAware = XInternAtom(m_display, "XdndAware", False);
    XdndPosition = XInternAtom(m_display, "XdndPosition", False);
    XdndStatus = XInternAtom(m_display, "XdndStatus", False);
    XdndActionCopy = XInternAtom(m_display, "XdndActionCopy", False);
    XdndDrop = XInternAtom(m_display, "XdndDrop", False);
    XdndFinished = XInternAtom(m_display, "XdndFinished", False);
    XdndSelection = XInternAtom(m_display, "XdndSelection", False);
    URI_LIST = XInternAtom(m_display, "text/uri-list", False);
  }

  Atom protocolVersion = 5;
  XChangeProperty(
    m_display, m_window,
    XdndAware, XA_ATOM, 32,
    PropModeReplace,
    (const unsigned char*)&protocolVersion, 1);

  WindowX11::addWindow(this);
}

WindowX11::~WindowX11()
{
  if (m_xic)
    XDestroyIC(m_xic);
  XFreeGC(m_display, m_gc);
  XDestroyWindow(m_display, m_window);

  WindowX11::removeWindow(this);
}

os::ScreenRef WindowX11::screen() const
{
  return os::make_ref<ScreenX11>(DefaultScreen(m_display));
}

os::ColorSpaceRef WindowX11::colorSpace() const
{
  if (auto defaultCS = os::instance()->windowsColorSpace())
    return defaultCS;

  // TODO get the window color space
  return os::instance()->makeColorSpace(gfx::ColorSpace::MakeSRGB());
}

void WindowX11::setScale(const int scale)
{
  m_scale = scale;

  // Adjust increment/decrement of the window to be multiplies of m_scale
  XSizeHints* hints = XAllocSizeHints();
  hints->flags  = PResizeInc;
  hints->width_inc = m_scale;
  hints->height_inc = m_scale;
  XSetWMNormalHints(m_display, m_window, hints);
  XFree(hints);

  onResize(clientSize());
}

bool WindowX11::isVisible() const
{
  // TODO
  return true;
}

void WindowX11::setVisible(bool visible)
{
  // TODO
}

void WindowX11::activate()
{
  const Atom _NET_ACTIVE_WINDOW = XInternAtom(m_display, "_NET_ACTIVE_WINDOW", False);
  if (!_NET_ACTIVE_WINDOW)
    return;                     // No atoms?

  ::Window root = XDefaultRootWindow(m_display);
  XEvent event;
  memset(&event, 0, sizeof(event));
  event.xany.type = ClientMessage;
  event.xclient.window = m_window;
  event.xclient.message_type = _NET_ACTIVE_WINDOW;
  event.xclient.format = 32;
  event.xclient.data.l[0] = 1; // 1 when the request comes from an application
  event.xclient.data.l[1] = CurrentTime;
  event.xclient.data.l[2] = 0;
  event.xclient.data.l[3] = 0;

  XSendEvent(m_display, root, 0,
             SubstructureNotifyMask | SubstructureRedirectMask, &event);
}

void WindowX11::maximize()
{
  ::Window root = XDefaultRootWindow(m_display);
  XEvent event;
  memset(&event, 0, sizeof(event));
  event.xany.type = ClientMessage;
  event.xclient.window = m_window;
  event.xclient.message_type = _NET_WM_STATE;
  event.xclient.format = 32;
  event.xclient.data.l[0] = (isMaximized() ? _NET_WM_STATE_REMOVE:
                                             _NET_WM_STATE_ADD);
  event.xclient.data.l[1] = _NET_WM_STATE_MAXIMIZED_VERT;
  event.xclient.data.l[2] = _NET_WM_STATE_MAXIMIZED_HORZ;

  XSendEvent(m_display, root, 0,
             SubstructureNotifyMask | SubstructureRedirectMask, &event);
}

void WindowX11::minimize()
{
  XIconifyWindow(m_display, m_window, DefaultScreen(m_display));
}

bool WindowX11::isMaximized() const
{
  bool result = false;
  Atom actual_type;
  int actual_format;
  unsigned long nitems;
  unsigned long bytes_after;
  Atom* prop = nullptr;
  const int res = XGetWindowProperty(
    m_display, m_window,
    _NET_WM_STATE,
    // TODO is 256 enough?
    0, 256,
    False, XA_ATOM,
    &actual_type, &actual_format,
    &nitems, &bytes_after,
    (unsigned char**)&prop);

  if (res == Success) {
    for (int i=0; i<nitems; ++i) {
      if (prop[i] == _NET_WM_STATE_MAXIMIZED_VERT ||
          prop[i] == _NET_WM_STATE_MAXIMIZED_HORZ) {
        result = true;
      }
    }
    XFree(prop);
  }
  return result;
}

bool WindowX11::isMinimized() const
{
  return false;
}

bool WindowX11::isTransparent() const
{
  return m_transparent;
}

bool WindowX11::isFullscreen() const
{
  // TODO ask _NET_WM_STATE_FULLSCREEN atom in _NET_WM_STATE window property
  return m_fullscreen;
}

void WindowX11::setFullscreen(bool state)
{
  if (isFullscreen() == state)
    return;

  const Atom _NET_WM_STATE_FULLSCREEN = XInternAtom(m_display, "_NET_WM_STATE_FULLSCREEN", False);
  if (!_NET_WM_STATE || !_NET_WM_STATE_FULLSCREEN)
    return;                     // No atoms?

  // From _NET_WM_STATE section in https://specifications.freedesktop.org/wm-spec/1.3/ar01s05.html#idm46018259875952
  //
  //   "Client wishing to change the state of a window MUST send a
  //    _NET_WM_STATE client message to the root window. The Window
  //    Manager MUST keep this property updated to reflect the
  //    current state of the window."
  //
  const ::Window root = XDefaultRootWindow(m_display);
  XEvent event;
  memset(&event, 0, sizeof(event));
  event.xany.type = ClientMessage;
  event.xclient.window = m_window;
  event.xclient.message_type = _NET_WM_STATE;
  event.xclient.format = 32;
  // The action
  event.xclient.data.l[0] = (state ? _NET_WM_STATE_ADD:
                                     _NET_WM_STATE_REMOVE);
  event.xclient.data.l[1] = _NET_WM_STATE_FULLSCREEN; // First property to alter
  event.xclient.data.l[2] = 0;      // Second property to alter
  event.xclient.data.l[3] = 0;      // Source indication

  XSendEvent(m_display, root, 0,
             SubstructureNotifyMask | SubstructureRedirectMask, &event);

  m_fullscreen = state;
}

void WindowX11::setTitle(const std::string& title)
{
  XTextProperty prop;
  prop.value = (unsigned char*)title.c_str();
  prop.encoding = XA_STRING;
  prop.format = 8;
  prop.nitems = std::strlen((char*)title.c_str());
  XSetWMName(m_display, m_window, &prop);
}

void WindowX11::setIcons(const SurfaceList& icons)
{
  if (!m_display || !m_window)
    return;

  bool first = true;
  for (const auto& icon : icons) {
    const int w = icon->width();
    const int h = icon->height();

    SurfaceFormatData format;
    icon->getFormat(&format);

    std::vector<unsigned long> data(w*h+2);
    int i = 0;
    data[i++] = w;
    data[i++] = h;
    for (int y=0; y<h; ++y) {
      const uint32_t* p = (const uint32_t*)icon->getData(0, y);
      for (int x=0; x<w; ++x, ++p) {
        const uint32_t c = *p;
        data[i++] =
          (((c & format.blueMask ) >> format.blueShift )      ) |
          (((c & format.greenMask) >> format.greenShift) <<  8) |
          (((c & format.redMask  ) >> format.redShift  ) << 16) |
          (((c & format.alphaMask) >> format.alphaShift) << 24);
      }
    }

    const Atom _NET_WM_ICON = XInternAtom(m_display, "_NET_WM_ICON", False);
    XChangeProperty(
      m_display, m_window, _NET_WM_ICON, XA_CARDINAL, 32,
      first ? PropModeReplace:
              PropModeAppend,
      (const unsigned char*)data.data(), data.size());

    first = false;
  }
}

gfx::Rect WindowX11::frame() const
{
  gfx::Rect rc = contentRect();
  if (!m_borderless)
    rc.enlarge(m_frameExtents);
  return rc;
}

void WindowX11::setFrame(const gfx::Rect& bounds)
{
  gfx::Rect rc = bounds;
  if (!m_borderless) {
    rc.shrink(m_frameExtents);
    rc.y -= m_frameExtents.top(); // Shrink from top
  }

  XMoveResizeWindow(
    m_display,
    m_window,
    rc.x, rc.y,
    rc.w, rc.h);
  XFlush(m_display);
}

gfx::Rect WindowX11::contentRect() const
{
  ::Window root;
  int x, y;
  unsigned int width, height, border, depth;
  XGetGeometry(m_display, m_window, &root,
               &x, &y, &width, &height, &border, &depth);

  ::Window child_return;
  XTranslateCoordinates(m_display, m_window, root,
                        0, 0, &x, &y, &child_return);

  return gfx::Rect(x, y, int(width), int(height));
}

std::string WindowX11::title() const
{
  XTextProperty prop;
  if (!XGetWMName(m_display, m_window, &prop) || !prop.value)
    return std::string();

  std::string value = (const char*)prop.value;
  XFree(prop.value);
  return value;
}

gfx::Size WindowX11::clientSize() const
{
  ::Window root;
  int x, y;
  unsigned int width, height, border, depth;
  XGetGeometry(m_display, m_window, &root,
               &x, &y, &width, &height, &border, &depth);
  return gfx::Size(int(width), int(height));
}

gfx::Rect WindowX11::restoredFrame() const
{
  ::Window root;
  int x, y;
  unsigned int width, height, border, depth;
  XGetGeometry(m_display, m_window, &root,
               &x, &y, &width, &height, &border, &depth);
  return gfx::Rect(x, y, int(width), int(height));
}

void WindowX11::captureMouse()
{
  XGrabPointer(m_display, m_window, False,
               PointerMotionMask | ButtonPressMask | ButtonReleaseMask,
               GrabModeAsync, GrabModeAsync,
               None, None, CurrentTime);
}

void WindowX11::releaseMouse()
{
  XUngrabPointer(m_display, CurrentTime);
}

void WindowX11::setMousePosition(const gfx::Point& position)
{
  ::Window root;
  int x, y;
  unsigned int w, h, border, depth;
  XGetGeometry(m_display, m_window, &root,
               &x, &y, &w, &h, &border, &depth);
  XWarpPointer(m_display, m_window, m_window, 0, 0, w, h,
               position.x*m_scale, position.y*m_scale);
}

void WindowX11::invalidateRegion(const gfx::Region& rgn)
{
  const gfx::Rect bounds = rgn.bounds();
  onPaint(gfx::Rect(bounds.x*m_scale,
                    bounds.y*m_scale,
                    bounds.w*m_scale,
                    bounds.h*m_scale));
}

bool WindowX11::setCursor(NativeCursor nativeCursor)
{
  const CursorRef cursor = ((SystemX11*)os::instance())->getNativeCursor(nativeCursor);
  if (cursor)
    return setX11Cursor((::Cursor)cursor->nativeHandle());
  return false;
}

bool WindowX11::setCursor(const CursorRef& cursor)
{
  ASSERT(cursor);
  if (!cursor)
    return false;

  if (cursor->nativeHandle())
    return setX11Cursor((::Cursor)cursor->nativeHandle());
  return setCursor(NativeCursor::Hidden);
}

void WindowX11::performWindowAction(const WindowAction action,
                                    const Event* ev)
{
  const Atom _NET_WM_MOVERESIZE = XInternAtom(m_display, "_NET_WM_MOVERESIZE", False);
  if (!_NET_WM_MOVERESIZE)
    return;                     // No atoms?

  int x, y;
  if (ev) {
    x = ev->position().x;
    y = ev->position().y;
  }
  else {
    int rootx, rooty;
    unsigned int mask;
    ::Window root, child;
    if (!XQueryPointer(m_display, m_window, &root, &child, &rootx, &rooty, &x, &y, &mask)) {
      x = 0;
      y = 0;
    }
  }

  const int button = (ev ? get_x_mouse_button_from_event(ev->button()): 0);
  Atom direction = 0;
  switch (action) {
    case WindowAction::Cancel:                direction = _NET_WM_MOVERESIZE_CANCEL; break;
    case WindowAction::Move:                  direction = _NET_WM_MOVERESIZE_MOVE; break;
    case WindowAction::ResizeFromTopLeft:     direction = _NET_WM_MOVERESIZE_SIZE_TOPLEFT; break;
    case WindowAction::ResizeFromTop:         direction = _NET_WM_MOVERESIZE_SIZE_TOP; break;
    case WindowAction::ResizeFromTopRight:    direction = _NET_WM_MOVERESIZE_SIZE_TOPRIGHT; break;
    case WindowAction::ResizeFromLeft:        direction = _NET_WM_MOVERESIZE_SIZE_LEFT; break;
    case WindowAction::ResizeFromRight:       direction = _NET_WM_MOVERESIZE_SIZE_RIGHT; break;
    case WindowAction::ResizeFromBottomLeft:  direction = _NET_WM_MOVERESIZE_SIZE_BOTTOMLEFT; break;
    case WindowAction::ResizeFromBottom:      direction = _NET_WM_MOVERESIZE_SIZE_BOTTOM; break;
    case WindowAction::ResizeFromBottomRight: direction = _NET_WM_MOVERESIZE_SIZE_BOTTOMRIGHT; break;
  }

  // From:
  // https://specifications.freedesktop.org/wm-spec/latest/ar01s04.html#idm46075117309248
  // "The Client MUST release all grabs prior to sending such
  //  message (except for the _NET_WM_MOVERESIZE_CANCEL message)."
  if (direction != _NET_WM_MOVERESIZE_CANCEL)
    releaseMouse();

  const ::Window root = XDefaultRootWindow(m_display);
  ::Window child;
  XTranslateCoordinates(m_display, m_window, root,
                        x, y, &x, &y, &child);

  XEvent event;
  memset(&event, 0, sizeof(event));
  event.xany.type = ClientMessage;
  event.xclient.window = m_window;
  event.xclient.message_type = _NET_WM_MOVERESIZE;
  event.xclient.format = 32;
  event.xclient.data.l[0] = x;
  event.xclient.data.l[1] = y;
  event.xclient.data.l[2] = direction;
  event.xclient.data.l[3] = button;
  event.xclient.data.l[4] = 0;

  XSendEvent(m_display, root, 0,
             SubstructureNotifyMask | SubstructureRedirectMask, &event);
}

void WindowX11::setWMClass(const std::string& res_class)
{
  const std::string res_name = base::string_to_lower(res_class);
  XClassHint ch;
  ch.res_name = (char*)res_name.c_str();
  ch.res_class = (char*)res_class.c_str();
  XSetClassHint(m_display, m_window, &ch);
}

// TODO this doesn't work on GNOME 3, so still some work is required
void WindowX11::setAllowedActions()
{
  Atom actual_type;
  int actual_format;
  unsigned long nitems;
  unsigned long bytes_after;
  Atom* prop = nullptr;
  const int res = XGetWindowProperty(
    m_display, m_window,
    _NET_WM_ALLOWED_ACTIONS,
    0, 256,
    False, XA_ATOM,
    &actual_type, &actual_format,
    &nitems, &bytes_after,
    (unsigned char**)&prop);
  if (res != Success)
    return;

  std::vector<Atom> allowed;
  for (int i=0; i<nitems; ++i)
    allowed.push_back(prop[i]);
  XFree(prop);

  // Auxiliary function to match one allowed action from the "spec"
  // with the specific Atom required in the _NET_WM_ALLOWED_ACTIONS
  // property.
  auto set_allowed_action =
    [&allowed, this](const bool expected, const char* atomName) {
      const Atom atom = XInternAtom(m_display, atomName, False);
      if (!atom)
        return;
      auto it = std::find(allowed.begin(), allowed.end(), atom);
      if (expected) {
        // Add missing atom/action
        if (it == allowed.end()) {
          allowed.push_back(atom);
        }
      }
      else {
        // Remove disallowed atom/action
        if (it != allowed.end()) {
          allowed.erase(it);
        }
      }
    };

  set_allowed_action(m_resizable, "_NET_WM_ACTION_RESIZE");
  set_allowed_action(m_resizable, "_NET_WM_ACTION_FULLSCREEN");
  set_allowed_action(m_minimizable, "_NET_WM_ACTION_MINIMIZE");
  set_allowed_action(m_maximizable, "_NET_WM_ACTION_MAXIMIZE_VERT");
  set_allowed_action(m_maximizable, "_NET_WM_ACTION_MAXIMIZE_HORZ");
  set_allowed_action(m_closable, "_NET_WM_ACTION_CLOSE");

  XChangeProperty(
    m_display, m_window, _NET_WM_ALLOWED_ACTIONS,
    XA_ATOM, 32, (nitems == 0 ? PropModeAppend:
                                PropModeReplace),
    (const unsigned char*)allowed.data(), allowed.size());
}

bool WindowX11::setX11Cursor(::Cursor xcursor)
{
  if (xcursor != None) {
    XDefineCursor(m_display, m_window, xcursor);
    return true;
  }
  return false;
}

bool WindowX11::requestX11FrameExtents()
{
  const ::Window root = XDefaultRootWindow(m_display);

  // Send a _NET_REQUEST_FRAME_EXTENTS to the root window to ask for
  // the frame extents of this window.
  XEvent event;
  memset(&event, 0, sizeof(event));
  event.xany.type = ClientMessage;
  event.xclient.window = m_window;
  event.xclient.message_type = _NET_REQUEST_FRAME_EXTENTS;
  event.xclient.format = 32;
  XSendEvent(m_display, root, 0,
             SubstructureNotifyMask | SubstructureRedirectMask, &event);

  // Now we have to wait the _NET_FRAME_EXTENTS property modification
  // event.
  auto isFrameExtentsEvent =
    [](Display* d, XEvent* e, XPointer w) -> Bool {
      return (e->xany.type == PropertyNotify &&
              e->xproperty.window == (::Window)w &&
              e->xproperty.atom == _NET_FRAME_EXTENTS);
    };

  XEvent event2;
  int wait = 100;        // We're going to wait 100 milliseconds
  while (wait > 0) {
    if (XCheckIfEvent(m_display,
                      &event2,
                      isFrameExtentsEvent,
                      (XPointer)m_window)) {
      // Event in queue (the event is not removed from the queue).
      return true;
    }
    base::this_thread::sleep_for(0.01);
    wait -= 10;
  }

  // We're not sure if we're going to receive the _NET_FRAME_EXTENTS
  // notification in the future
  return false;
}

void WindowX11::getX11FrameExtents()
{
  Atom actual_type;
  int actual_format;
  unsigned long nitems;
  unsigned long bytes_after;
  unsigned long* prop = nullptr;
  const int res = XGetWindowProperty(
    m_display, m_window,
    _NET_FRAME_EXTENTS,
    0, 4,
    False, XA_CARDINAL,
    &actual_type, &actual_format,
    &nitems, &bytes_after,
    (unsigned char**)&prop);

  if (res == Success && nitems == 4) {
    // Get the dimension of the title bar + borders (WM decorators)
    m_frameExtents.left(prop[0]);
    m_frameExtents.right(prop[1]);
    m_frameExtents.top(prop[2]);
    m_frameExtents.bottom(prop[3]);
    XFree(prop);
  }
}

void WindowX11::processX11Event(XEvent& event)
{
  auto* xinput = X11::instance()->xinput();
  if (xinput->handleExtensionEvent(event)) {
    Event ev;
    xinput->convertExtensionEvent(event, ev, m_scale,
                                  g_lastXInputEventTime);
    queueEvent(ev);
    return;
  }

  switch (event.type) {

    case ConfigureNotify: {
      const gfx::Rect rc(event.xconfigure.x, event.xconfigure.y,
                         event.xconfigure.width, event.xconfigure.height);

      if (rc.w > 0 && rc.h > 0 && rc.size() != m_lastClientSize) {
        m_lastClientSize = rc.size();
        onResize(rc.size());
      }
      break;
    }

    case Expose: {
      const gfx::Rect rc(event.xexpose.x, event.xexpose.y,
                         event.xexpose.width, event.xexpose.height);
      onPaint(rc);
      break;
    }

    case KeyPress:
    case KeyRelease: {
      Event ev;
      ev.setType(event.type == KeyPress ? Event::KeyDown: Event::KeyUp);

      const KeySym keysym = XLookupKeysym(&event.xkey, 0);
      ev.setScancode(x11_keysym_to_scancode(keysym));

      if (m_xic) {
        std::vector<char> buf(16);
        const size_t len = Xutf8LookupString(m_xic, &event.xkey,
                                             buf.data(), buf.size(),
                                             nullptr, nullptr);
        if (len < buf.size())
          buf[len] = 0;
        std::wstring wideChars = base::from_utf8(std::string(buf.data()));
        if (!wideChars.empty())
          ev.setUnicodeChar(wideChars[0]);
        KEY_TRACE("Xutf8LookupString %s\n", &buf[0]);
      }


      // Check if the key has been pressed, and if yes - check what's
      // the previous state of the key symbol - if it was previously
      // pressed set repeat count of 1, if it wasn't pressed previously
      // it will set repeat count to 0
      if (event.type == KeyPress) {
        if (g_pressedKeys.find(keysym) != g_pressedKeys.end())
          ev.setRepeat(1);
        else
          g_pressedKeys.insert(keysym);
      }
      else
        g_pressedKeys.erase(keysym);

      // Key event used by the input method (e.g. when the users
      // presses a dead key).
      if (XFilterEvent(&event, m_window))
        break;

      int modifiers = (int)get_modifiers_from_x(event.xkey.state);
      switch (keysym) {
        case XK_space: {
          switch (event.type) {
            case KeyPress:
              g_spaceBarIsPressed = true;
              break;
            case KeyRelease:
              g_spaceBarIsPressed = false;

              // If the next event after a KeyRelease is a KeyPress of
              // the same keycode (the space bar in this case), it
              // means that this KeyRelease is just a repetition of a
              // the same keycode.
              if (XEventsQueued(m_display, QueuedAfterReading)) {
                XEvent nextEvent;
                XPeekEvent(m_display, &nextEvent);
                if (nextEvent.type == KeyPress &&
                    nextEvent.xkey.time == event.xkey.time &&
                    nextEvent.xkey.keycode == event.xkey.keycode) {
                  g_spaceBarIsPressed = true;
                }
              }
              break;
          }
          break;
        }
        case XK_Shift_L:
        case XK_Shift_R:
          if (event.type == KeyPress)
            modifiers |= kKeyShiftModifier;
          else
            modifiers &= ~kKeyShiftModifier;
          break;
        case XK_Control_L:
        case XK_Control_R:
          if (event.type == KeyPress)
            modifiers |= kKeyCtrlModifier;
          else
            modifiers &= ~kKeyCtrlModifier;
          break;
        case XK_Alt_L:
        case XK_Alt_R:
          if (event.type == KeyPress)
            modifiers |= kKeyAltModifier;
          else
            modifiers &= ~kKeyAltModifier;
          break;
        case XK_Meta_L:
        case XK_Super_L:
        case XK_Meta_R:
        case XK_Super_R:
          if (event.type == KeyPress)
            modifiers |= kKeyWinModifier;
          else
            modifiers &= ~kKeyWinModifier;
          break;
      }
      ev.setModifiers((KeyModifiers)modifiers);
      KEY_TRACE("%s state=%04x keycode=%04x\n",
                (event.type == KeyPress ? "KeyPress": "KeyRelease"),
                event.xkey.state,
                event.xkey.keycode);
      KEY_TRACE(" > %s\n", XKeysymToString(keysym));

      queueEvent(ev);
      break;
    }

    case ButtonPress:
    case ButtonRelease: {
      // This can happen when the button press/release events are
      // handled in XInput
      if (event.xmotion.time == g_lastXInputEventTime)
        break;

      Event ev;
      if (is_mouse_wheel_button(event.xbutton.button)) {
        if (event.type == ButtonPress) {
          ev.setType(Event::MouseWheel);
          ev.setWheelDelta(get_mouse_wheel_delta(event.xbutton.button));
        }
        else {
          // Ignore ButtonRelese for the mouse wheel to avoid
          // duplicating MouseWheel event effects.
          break;
        }
      }
      else {
        ev.setType(event.type == ButtonPress ? Event::MouseDown:
                                               Event::MouseUp);

        const Event::MouseButton button =
          get_mouse_button_from_x(event.xbutton.button);
        ev.setButton(button);

        if (event.type == ButtonPress) {
          if (m_doubleClickButton == button &&
              base::current_tick() - m_doubleClickTick < LAF_X11_DOUBLE_CLICK_TIMEOUT) {
            ev.setType(Event::MouseDoubleClick);
            m_doubleClickButton = Event::NoneButton;
          }
          else {
            m_doubleClickButton = button;
            m_doubleClickTick = base::current_tick();
          }
        }
      }
      ev.setModifiers(get_modifiers_from_x(event.xbutton.state));
      ev.setPosition(gfx::Point(event.xbutton.x / m_scale,
                                event.xbutton.y / m_scale));

      queueEvent(ev);
      break;
    }

    case MotionNotify: {
      // This can happen when the motion event are handled in XInput
      if (event.xmotion.time == g_lastXInputEventTime)
        break;

      // Reset double-click state
      m_doubleClickButton = Event::NoneButton;

      const gfx::Point pos(event.xmotion.x / m_scale,
                           event.xmotion.y / m_scale);

      if (m_lastMousePos == pos)
        break;
      m_lastMousePos = pos;

      Event ev;
      ev.setType(Event::MouseMove);
      ev.setModifiers(get_modifiers_from_x(event.xmotion.state));
      ev.setPosition(pos);
      queueEvent(ev);
      break;
    }

    case EnterNotify:
    case LeaveNotify:
      g_spaceBarIsPressed = false;

      // "mode" can be NotifyGrab or NotifyUngrab when middle mouse
      // button is pressed/released. We must not generated
      // MouseEnter/Leave events on those cases, only on NotifyNormal
      // (when mouse leaves/enter the X11 window).
      if (event.xcrossing.mode == NotifyNormal) {
        Event ev;
        ev.setType(event.type == EnterNotify ? Event::MouseEnter:
                                               Event::MouseLeave);
        ev.setModifiers(get_modifiers_from_x(event.xcrossing.state));
        ev.setPosition(gfx::Point(event.xcrossing.x / m_scale,
                                  event.xcrossing.y / m_scale));
        queueEvent(ev);
      }
      break;

    case ClientMessage:
      // When the close button is pressed
      if (event.xclient.message_type == WM_PROTOCOLS &&
          Atom(event.xclient.data.l[0]) == WM_DELETE_WINDOW) {
        Event ev;
        ev.setType(Event::CloseWindow);
        queueEvent(ev);
      }
      else if (event.xclient.message_type == XdndPosition) {
        auto sourceWindow = (::Window)event.xclient.data.l[0];
        // Save the latest mouse position reported by the source window
        g_dndPosition.x = event.xclient.data.l[2] >> 16;
        g_dndPosition.y = event.xclient.data.l[2] & 0xFFFF;

        // TODO Ask to the library user if we can drop and the action
        //      that will take place
        XEvent event2;
        memset(&event2, 0, sizeof(event2));
        event2.xany.type = ClientMessage;
        event2.xclient.window = sourceWindow;
        event2.xclient.message_type = XdndStatus;
        event2.xclient.format = 32;
        event2.xclient.data.l[0] = m_window;
        // Bit 0 = this window accept the drop
        event2.xclient.data.l[1] = 1;
        event2.xclient.data.l[4] = XdndActionCopy;
        XSendEvent(m_display, sourceWindow, 0, 0, &event2);
      }
      else if (event.xclient.message_type == XdndDrop) {
        // Save the X11 window from where this XdndDrop message came
        // from, so then we can send a XdndFinished later.
        g_dndSource = (::Window)event.xclient.data.l[0];

        // Ask for the XdndSelection, we're going to receive the
        // dropped items in the SelectionNotify.
        XConvertSelection(m_display, XdndSelection,
                          URI_LIST, XdndSelection,
                          m_window, CurrentTime);
      }
      break;

    case SelectionNotify:
      if (event.xselection.property == XdndSelection) {
        bool successful = false;
        Atom actual_type;
        int actual_format;
        unsigned long nitems;
        unsigned long bytes_after;
        char* prop = nullptr;
        const int res = XGetWindowProperty(
          m_display,
          m_window,
          XdndSelection,
          0, 256,
          False, URI_LIST,
          &actual_type, &actual_format,
          &nitems, &bytes_after,
          (unsigned char**)&prop);

        if (prop) {
          if (actual_type == URI_LIST) {
            std::vector<std::string> files;
            base::split_string(std::string(prop), files, "\n");
            for (auto it=files.begin(); it!=files.end(); ) {
              const std::string f = decode_url(*it);
              if (f.empty())
                it = files.erase(it);
              else {
                *it = f;
                ++it;
              }
            }

            if (!files.empty()) {
              os::Event ev;
              ev.setType(os::Event::DropFiles);
              ev.setFiles(files);
              // Mouse position is relative to the root window, so
              // we make it relative to the content rect.
              ev.setPosition(g_dndPosition - contentRect().origin());
              queueEvent(ev);

              successful = true;
            }
          }

          XFree(prop);
        }

        const ::Window root = XDefaultRootWindow(m_display);
        XEvent event2;
        memset(&event2, 0, sizeof(event2));
        event2.xany.type = ClientMessage;
        event2.xclient.window = g_dndSource;
        event2.xclient.message_type = XdndFinished;
        event2.xclient.format = 32;
        event2.xclient.data.l[0] = m_window;
        // Set bit 0 when the drop operation was accepted.
        event2.xclient.data.l[1] = (successful ? 1: 0);
        event2.xclient.data.l[2] = 0;
        event2.xclient.data.l[3] = 0;
        XSendEvent(m_display, root, 0, 0, &event2);
      }
      break;

    case PropertyNotify:
      if (event.xproperty.atom == _NET_FRAME_EXTENTS) {
        getX11FrameExtents();

        if (m_borderless && m_frameExtents != gfx::Border(0, 0, 0, 0)) {
          std::vector<unsigned long> data(4, 0);
          XChangeProperty(
            m_display, m_window, _NET_FRAME_EXTENTS, XA_CARDINAL, 32,
            PropModeReplace, (const unsigned char*)data.data(), data.size());
        }
      }
      else if (event.xproperty.atom == _NET_WM_ALLOWED_ACTIONS) {
        // Set allowed actions (resize, maximize, etc.)
        if (!m_initializingActions) {
          m_initializingActions = true;
          setAllowedActions();
        }
      }
      break;
  }
}

} // namespace os
