/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.base.VerifyException;
import io.grpc.Attributes;
import io.grpc.EquivalentAddressGroup;
import io.grpc.NameResolver;
import io.grpc.ProxiedSocketAddress;
import io.grpc.ProxyDetector;
import io.grpc.Status;
import io.grpc.SynchronizationContext;
import io.grpc.internal.GrpcAttributes;
import io.grpc.internal.JsonParser;
import io.grpc.internal.ServiceConfigUtil;
import io.grpc.internal.SharedResourceHolder;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;

final class DnsNameResolver
extends NameResolver {
    private static final Logger logger = Logger.getLogger(DnsNameResolver.class.getName());
    private static final String SERVICE_CONFIG_CHOICE_CLIENT_LANGUAGE_KEY = "clientLanguage";
    private static final String SERVICE_CONFIG_CHOICE_PERCENTAGE_KEY = "percentage";
    private static final String SERVICE_CONFIG_CHOICE_CLIENT_HOSTNAME_KEY = "clientHostname";
    private static final String SERVICE_CONFIG_CHOICE_SERVICE_CONFIG_KEY = "serviceConfig";
    static final String SERVICE_CONFIG_PREFIX = "grpc_config=";
    private static final Set<String> SERVICE_CONFIG_CHOICE_KEYS = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("clientLanguage", "percentage", "clientHostname", "serviceConfig")));
    private static final String SERVICE_CONFIG_NAME_PREFIX = "_grpc_config.";
    private static final String GRPCLB_NAME_PREFIX = "_grpclb._tcp.";
    private static final String JNDI_PROPERTY = System.getProperty("io.grpc.internal.DnsNameResolverProvider.enable_jndi", "true");
    private static final String JNDI_LOCALHOST_PROPERTY = System.getProperty("io.grpc.internal.DnsNameResolverProvider.enable_jndi_localhost", "false");
    private static final String JNDI_SRV_PROPERTY = System.getProperty("io.grpc.internal.DnsNameResolverProvider.enable_grpclb", "false");
    private static final String JNDI_TXT_PROPERTY = System.getProperty("io.grpc.internal.DnsNameResolverProvider.enable_service_config", "false");
    @VisibleForTesting
    static final String NETWORKADDRESS_CACHE_TTL_PROPERTY = "networkaddress.cache.ttl";
    @VisibleForTesting
    static final long DEFAULT_NETWORK_CACHE_TTL_SECONDS = 30L;
    @VisibleForTesting
    static boolean enableJndi = Boolean.parseBoolean(JNDI_PROPERTY);
    @VisibleForTesting
    static boolean enableJndiLocalhost = Boolean.parseBoolean(JNDI_LOCALHOST_PROPERTY);
    @VisibleForTesting
    static boolean enableSrv = Boolean.parseBoolean(JNDI_SRV_PROPERTY);
    @VisibleForTesting
    static boolean enableTxt = Boolean.parseBoolean(JNDI_TXT_PROPERTY);
    private static final ResourceResolverFactory resourceResolverFactory = DnsNameResolver.getResourceResolverFactory(DnsNameResolver.class.getClassLoader());
    @VisibleForTesting
    final ProxyDetector proxyDetector;
    private static String localHostname;
    private final Random random = new Random();
    private volatile AddressResolver addressResolver = JdkAddressResolver.INSTANCE;
    private final AtomicReference<ResourceResolver> resourceResolver = new AtomicReference();
    private final String authority;
    private final String host;
    private final int port;
    private final SharedResourceHolder.Resource<Executor> executorResource;
    private final long cacheTtlNanos;
    private final SynchronizationContext syncContext;
    private final Stopwatch stopwatch;
    private ResolutionResults cachedResolutionResults;
    private boolean shutdown;
    private Executor executor;
    private boolean resolving;
    private NameResolver.Listener listener;

    DnsNameResolver(@Nullable String nsAuthority, String name, NameResolver.Helper helper, SharedResourceHolder.Resource<Executor> executorResource, Stopwatch stopwatch, boolean isAndroid) {
        Preconditions.checkNotNull((Object)helper, (Object)"helper");
        this.executorResource = executorResource;
        URI nameUri = URI.create("//" + (String)Preconditions.checkNotNull((Object)name, (Object)"name"));
        Preconditions.checkArgument((nameUri.getHost() != null ? 1 : 0) != 0, (String)"Invalid DNS name: %s", (Object)name);
        this.authority = (String)Preconditions.checkNotNull((Object)nameUri.getAuthority(), (String)"nameUri (%s) doesn't have an authority", (Object)nameUri);
        this.host = nameUri.getHost();
        this.port = nameUri.getPort() == -1 ? helper.getDefaultPort() : nameUri.getPort();
        this.proxyDetector = (ProxyDetector)Preconditions.checkNotNull((Object)helper.getProxyDetector(), (Object)"proxyDetector");
        this.cacheTtlNanos = DnsNameResolver.getNetworkAddressCacheTtlNanos(isAndroid);
        this.stopwatch = (Stopwatch)Preconditions.checkNotNull((Object)stopwatch, (Object)"stopwatch");
        this.syncContext = (SynchronizationContext)Preconditions.checkNotNull((Object)helper.getSynchronizationContext(), (Object)"syncContext");
    }

    @Override
    public String getServiceAuthority() {
        return this.authority;
    }

    @Override
    public void start(NameResolver.Listener listener) {
        Preconditions.checkState((this.listener == null ? 1 : 0) != 0, (Object)"already started");
        this.executor = SharedResourceHolder.get(this.executorResource);
        this.listener = (NameResolver.Listener)Preconditions.checkNotNull((Object)listener, (Object)"listener");
        this.resolve();
    }

    @Override
    public void refresh() {
        Preconditions.checkState((this.listener != null ? 1 : 0) != 0, (Object)"not started");
        this.resolve();
    }

    @Nullable
    static NameResolver.Helper.ConfigOrError parseServiceConfig(List<String> rawTxtRecords, Random random, String localHostname) {
        List<Map<String, ?>> possibleServiceConfigChoices;
        try {
            possibleServiceConfigChoices = DnsNameResolver.parseTxtResults(rawTxtRecords);
        }
        catch (IOException | RuntimeException e) {
            return NameResolver.Helper.ConfigOrError.fromError(Status.UNKNOWN.withDescription("failed to parse TXT records").withCause(e));
        }
        Map<String, ?> possibleServiceConfig = null;
        for (Map<String, ?> possibleServiceConfigChoice : possibleServiceConfigChoices) {
            try {
                possibleServiceConfig = DnsNameResolver.maybeChooseServiceConfig(possibleServiceConfigChoice, random, localHostname);
            }
            catch (RuntimeException e) {
                return NameResolver.Helper.ConfigOrError.fromError(Status.UNKNOWN.withDescription("failed to pick service config choice").withCause(e));
            }
            if (possibleServiceConfig == null) continue;
            break;
        }
        if (possibleServiceConfig == null) {
            return null;
        }
        return NameResolver.Helper.ConfigOrError.fromConfig(possibleServiceConfig);
    }

    private void resolve() {
        if (this.resolving || this.shutdown || !this.cacheRefreshRequired()) {
            return;
        }
        this.resolving = true;
        this.executor.execute(new Resolve(this.listener));
    }

    private boolean cacheRefreshRequired() {
        return this.cachedResolutionResults == null || this.cacheTtlNanos == 0L || this.cacheTtlNanos > 0L && this.stopwatch.elapsed(TimeUnit.NANOSECONDS) > this.cacheTtlNanos;
    }

    @Override
    public void shutdown() {
        if (this.shutdown) {
            return;
        }
        this.shutdown = true;
        if (this.executor != null) {
            this.executor = SharedResourceHolder.release(this.executorResource, this.executor);
        }
    }

    final int getPort() {
        return this.port;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    static ResolutionResults resolveAll(AddressResolver addressResolver, @Nullable ResourceResolver resourceResolver, boolean requestSrvRecords, boolean requestTxtRecords, String name) {
        List<Object> addresses = Collections.emptyList();
        Exception addressesException = null;
        List<EquivalentAddressGroup> balancerAddresses = Collections.emptyList();
        Exception balancerAddressesException = null;
        List<String> txtRecords = Collections.emptyList();
        Exception txtRecordsException = null;
        try {
            addresses = addressResolver.resolveAddress(name);
        }
        catch (Exception e) {
            addressesException = e;
        }
        if (resourceResolver != null) {
            if (requestSrvRecords) {
                try {
                    balancerAddresses = resourceResolver.resolveSrv(addressResolver, GRPCLB_NAME_PREFIX + name);
                }
                catch (Exception e) {
                    balancerAddressesException = e;
                }
            }
            if (requestTxtRecords) {
                boolean dontResolveTxt;
                boolean balancerLookupFailedOrNotAttempted = !requestSrvRecords || balancerAddressesException != null;
                boolean bl = dontResolveTxt = addressesException != null && balancerLookupFailedOrNotAttempted;
                if (!dontResolveTxt) {
                    try {
                        txtRecords = resourceResolver.resolveTxt(SERVICE_CONFIG_NAME_PREFIX + name);
                    }
                    catch (Exception e) {
                        txtRecordsException = e;
                    }
                }
            }
        }
        try {
            if (addressesException != null && (balancerAddressesException != null || balancerAddresses.isEmpty())) {
                Throwables.throwIfUnchecked((Throwable)addressesException);
                throw new RuntimeException(addressesException);
            }
        }
        finally {
            if (addressesException != null) {
                logger.log(Level.FINE, "Address resolution failure", addressesException);
            }
            if (balancerAddressesException != null) {
                logger.log(Level.FINE, "Balancer resolution failure", balancerAddressesException);
            }
            if (txtRecordsException != null) {
                logger.log(Level.FINE, "ServiceConfig resolution failure", txtRecordsException);
            }
        }
        return new ResolutionResults(addresses, txtRecords, balancerAddresses);
    }

    @VisibleForTesting
    static List<Map<String, ?>> parseTxtResults(List<String> txtRecords) throws IOException {
        ArrayList possibleServiceConfigChoices = new ArrayList();
        for (String txtRecord : txtRecords) {
            if (!txtRecord.startsWith(SERVICE_CONFIG_PREFIX)) {
                logger.log(Level.FINE, "Ignoring non service config {0}", new Object[]{txtRecord});
                continue;
            }
            Object rawChoices = JsonParser.parse(txtRecord.substring(SERVICE_CONFIG_PREFIX.length()));
            if (!(rawChoices instanceof List)) {
                throw new ClassCastException("wrong type " + rawChoices);
            }
            List listChoices = (List)rawChoices;
            possibleServiceConfigChoices.addAll(ServiceConfigUtil.checkObjectList(listChoices));
        }
        return possibleServiceConfigChoices;
    }

    @Nullable
    private static final Double getPercentageFromChoice(Map<String, ?> serviceConfigChoice) {
        if (!serviceConfigChoice.containsKey(SERVICE_CONFIG_CHOICE_PERCENTAGE_KEY)) {
            return null;
        }
        return ServiceConfigUtil.getDouble(serviceConfigChoice, SERVICE_CONFIG_CHOICE_PERCENTAGE_KEY);
    }

    @Nullable
    private static final List<String> getClientLanguagesFromChoice(Map<String, ?> serviceConfigChoice) {
        if (!serviceConfigChoice.containsKey(SERVICE_CONFIG_CHOICE_CLIENT_LANGUAGE_KEY)) {
            return null;
        }
        return ServiceConfigUtil.checkStringList(ServiceConfigUtil.getList(serviceConfigChoice, SERVICE_CONFIG_CHOICE_CLIENT_LANGUAGE_KEY));
    }

    @Nullable
    private static final List<String> getHostnamesFromChoice(Map<String, ?> serviceConfigChoice) {
        if (!serviceConfigChoice.containsKey(SERVICE_CONFIG_CHOICE_CLIENT_HOSTNAME_KEY)) {
            return null;
        }
        return ServiceConfigUtil.checkStringList(ServiceConfigUtil.getList(serviceConfigChoice, SERVICE_CONFIG_CHOICE_CLIENT_HOSTNAME_KEY));
    }

    private static long getNetworkAddressCacheTtlNanos(boolean isAndroid) {
        if (isAndroid) {
            return 0L;
        }
        String cacheTtlPropertyValue = System.getProperty(NETWORKADDRESS_CACHE_TTL_PROPERTY);
        long cacheTtl = 30L;
        if (cacheTtlPropertyValue != null) {
            try {
                cacheTtl = Long.parseLong(cacheTtlPropertyValue);
            }
            catch (NumberFormatException e) {
                logger.log(Level.WARNING, "Property({0}) valid is not valid number format({1}), fall back to default({2})", new Object[]{NETWORKADDRESS_CACHE_TTL_PROPERTY, cacheTtlPropertyValue, cacheTtl});
            }
        }
        return cacheTtl > 0L ? TimeUnit.SECONDS.toNanos(cacheTtl) : cacheTtl;
    }

    @Nullable
    @VisibleForTesting
    static Map<String, ?> maybeChooseServiceConfig(Map<String, ?> choice, Random random, String hostname) {
        Map<String, ?> sc;
        List<String> clientHostnames;
        Double percentage;
        for (Map.Entry<String, ?> entry : choice.entrySet()) {
            Verify.verify((boolean)SERVICE_CONFIG_CHOICE_KEYS.contains(entry.getKey()), (String)"Bad key: %s", entry);
        }
        List<String> clientLanguages = DnsNameResolver.getClientLanguagesFromChoice(choice);
        if (clientLanguages != null && !clientLanguages.isEmpty()) {
            boolean javaPresent = false;
            for (String lang : clientLanguages) {
                if (!"java".equalsIgnoreCase(lang)) continue;
                javaPresent = true;
                break;
            }
            if (!javaPresent) {
                return null;
            }
        }
        if ((percentage = DnsNameResolver.getPercentageFromChoice(choice)) != null) {
            int pct = percentage.intValue();
            Verify.verify((pct >= 0 && pct <= 100 ? 1 : 0) != 0, (String)"Bad percentage: %s", (Object)percentage);
            if (random.nextInt(100) >= pct) {
                return null;
            }
        }
        if ((clientHostnames = DnsNameResolver.getHostnamesFromChoice(choice)) != null && !clientHostnames.isEmpty()) {
            boolean hostnamePresent = false;
            for (String clientHostname : clientHostnames) {
                if (!clientHostname.equals(hostname)) continue;
                hostnamePresent = true;
                break;
            }
            if (!hostnamePresent) {
                return null;
            }
        }
        if ((sc = ServiceConfigUtil.getObject(choice, SERVICE_CONFIG_CHOICE_SERVICE_CONFIG_KEY)) == null) {
            throw new VerifyException(String.format("key '%s' missing in '%s'", choice, SERVICE_CONFIG_CHOICE_SERVICE_CONFIG_KEY));
        }
        return sc;
    }

    @VisibleForTesting
    void setAddressResolver(AddressResolver addressResolver) {
        this.addressResolver = addressResolver;
    }

    @VisibleForTesting
    void setResourceResolver(ResourceResolver resourceResolver) {
        this.resourceResolver.set(resourceResolver);
    }

    @Nullable
    private ResourceResolver getResourceResolver() {
        ResourceResolver rr = this.resourceResolver.get();
        if (rr == null && resourceResolverFactory != null) {
            assert (resourceResolverFactory.unavailabilityCause() == null);
            rr = resourceResolverFactory.newResourceResolver();
        }
        return rr;
    }

    @Nullable
    @VisibleForTesting
    static ResourceResolverFactory getResourceResolverFactory(ClassLoader loader) {
        ResourceResolverFactory rrf;
        Constructor<ResourceResolverFactory> jndiCtor;
        Class<ResourceResolverFactory> jndiClazz;
        try {
            jndiClazz = Class.forName("io.grpc.internal.JndiResourceResolverFactory", true, loader).asSubclass(ResourceResolverFactory.class);
        }
        catch (ClassNotFoundException e) {
            logger.log(Level.FINE, "Unable to find JndiResourceResolverFactory, skipping.", e);
            return null;
        }
        try {
            jndiCtor = jndiClazz.getConstructor(new Class[0]);
        }
        catch (Exception e) {
            logger.log(Level.FINE, "Can't find JndiResourceResolverFactory ctor, skipping.", e);
            return null;
        }
        try {
            rrf = jndiCtor.newInstance(new Object[0]);
        }
        catch (Exception e) {
            logger.log(Level.FINE, "Can't construct JndiResourceResolverFactory, skipping.", e);
            return null;
        }
        if (rrf.unavailabilityCause() != null) {
            logger.log(Level.FINE, "JndiResourceResolverFactory not available, skipping.", rrf.unavailabilityCause());
            return null;
        }
        return rrf;
    }

    private static String getLocalHostname() {
        if (localHostname == null) {
            try {
                localHostname = InetAddress.getLocalHost().getHostName();
            }
            catch (UnknownHostException e) {
                throw new RuntimeException(e);
            }
        }
        return localHostname;
    }

    @VisibleForTesting
    static boolean shouldUseJndi(boolean jndiEnabled, boolean jndiLocalhostEnabled, String target) {
        if (!jndiEnabled) {
            return false;
        }
        if ("localhost".equalsIgnoreCase(target)) {
            return jndiLocalhostEnabled;
        }
        if (target.contains(":")) {
            return false;
        }
        boolean alldigits = true;
        for (int i = 0; i < target.length(); ++i) {
            char c = target.charAt(i);
            if (c == '.') continue;
            alldigits &= c >= '0' && c <= '9';
        }
        return !alldigits;
    }

    static interface ResourceResolver {
        public List<String> resolveTxt(String var1) throws Exception;

        public List<EquivalentAddressGroup> resolveSrv(AddressResolver var1, String var2) throws Exception;
    }

    private static enum JdkAddressResolver implements AddressResolver
    {
        INSTANCE;


        @Override
        public List<InetAddress> resolveAddress(String host) throws UnknownHostException {
            return Collections.unmodifiableList(Arrays.asList(InetAddress.getAllByName(host)));
        }
    }

    static interface AddressResolver {
        public List<InetAddress> resolveAddress(String var1) throws Exception;
    }

    static interface ResourceResolverFactory {
        @Nullable
        public ResourceResolver newResourceResolver();

        @Nullable
        public Throwable unavailabilityCause();
    }

    @VisibleForTesting
    static final class ResolutionResults {
        final List<? extends InetAddress> addresses;
        final List<String> txtRecords;
        final List<EquivalentAddressGroup> balancerAddresses;

        ResolutionResults(List<? extends InetAddress> addresses, List<String> txtRecords, List<EquivalentAddressGroup> balancerAddresses) {
            this.addresses = Collections.unmodifiableList((List)Preconditions.checkNotNull(addresses, (Object)"addresses"));
            this.txtRecords = Collections.unmodifiableList((List)Preconditions.checkNotNull(txtRecords, (Object)"txtRecords"));
            this.balancerAddresses = Collections.unmodifiableList((List)Preconditions.checkNotNull(balancerAddresses, (Object)"balancerAddresses"));
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("addresses", this.addresses).add("txtRecords", this.txtRecords).add("balancerAddresses", this.balancerAddresses).toString();
        }
    }

    private final class Resolve
    implements Runnable {
        private final NameResolver.Listener savedListener;

        Resolve(NameResolver.Listener savedListener) {
            this.savedListener = (NameResolver.Listener)Preconditions.checkNotNull((Object)savedListener, (Object)"savedListener");
        }

        @Override
        public void run() {
            if (logger.isLoggable(Level.FINER)) {
                logger.finer("Attempting DNS resolution of " + DnsNameResolver.this.host);
            }
            try {
                this.resolveInternal();
                DnsNameResolver.this.syncContext.execute(new Runnable(){

                    @Override
                    public void run() {
                        DnsNameResolver.this.resolving = false;
                    }
                });
            }
            catch (Throwable throwable) {
                DnsNameResolver.this.syncContext.execute(new /* invalid duplicate definition of identical inner class */);
                throw throwable;
            }
        }

        @VisibleForTesting
        void resolveInternal() {
            ResolutionResults resolutionResults;
            ProxiedSocketAddress proxiedAddr;
            InetSocketAddress destination = InetSocketAddress.createUnresolved(DnsNameResolver.this.host, DnsNameResolver.this.port);
            try {
                proxiedAddr = DnsNameResolver.this.proxyDetector.proxyFor(destination);
            }
            catch (IOException e) {
                this.savedListener.onError(Status.UNAVAILABLE.withDescription("Unable to resolve host " + DnsNameResolver.this.host).withCause(e));
                return;
            }
            if (proxiedAddr != null) {
                if (logger.isLoggable(Level.FINER)) {
                    logger.finer("Using proxy address " + proxiedAddr);
                }
                EquivalentAddressGroup server = new EquivalentAddressGroup(proxiedAddr);
                this.savedListener.onAddresses(Collections.singletonList(server), Attributes.EMPTY);
                return;
            }
            try {
                ResourceResolver resourceResolver = null;
                if (DnsNameResolver.shouldUseJndi(enableJndi, enableJndiLocalhost, DnsNameResolver.this.host)) {
                    resourceResolver = DnsNameResolver.this.getResourceResolver();
                }
                final ResolutionResults results = DnsNameResolver.resolveAll(DnsNameResolver.this.addressResolver, resourceResolver, enableSrv, enableTxt, DnsNameResolver.this.host);
                resolutionResults = results;
                DnsNameResolver.this.syncContext.execute(new Runnable(){

                    @Override
                    public void run() {
                        DnsNameResolver.this.cachedResolutionResults = results;
                        if (DnsNameResolver.this.cacheTtlNanos > 0L) {
                            DnsNameResolver.this.stopwatch.reset().start();
                        }
                    }
                });
                if (logger.isLoggable(Level.FINER)) {
                    logger.finer("Found DNS results " + resolutionResults + " for " + DnsNameResolver.this.host);
                }
            }
            catch (Exception e) {
                this.savedListener.onError(Status.UNAVAILABLE.withDescription("Unable to resolve host " + DnsNameResolver.this.host).withCause(e));
                return;
            }
            ArrayList<EquivalentAddressGroup> servers = new ArrayList<EquivalentAddressGroup>();
            for (InetAddress inetAddress : resolutionResults.addresses) {
                servers.add(new EquivalentAddressGroup(new InetSocketAddress(inetAddress, DnsNameResolver.this.port)));
            }
            servers.addAll(resolutionResults.balancerAddresses);
            if (servers.isEmpty()) {
                this.savedListener.onError(Status.UNAVAILABLE.withDescription("No DNS backend or balancer addresses found for " + DnsNameResolver.this.host));
                return;
            }
            Attributes.Builder attrs = Attributes.newBuilder();
            if (!resolutionResults.txtRecords.isEmpty()) {
                NameResolver.Helper.ConfigOrError configOrError = DnsNameResolver.parseServiceConfig(resolutionResults.txtRecords, DnsNameResolver.this.random, DnsNameResolver.getLocalHostname());
                if (configOrError != null) {
                    if (configOrError.getError() != null) {
                        this.savedListener.onError(configOrError.getError());
                        return;
                    }
                    Map config = (Map)configOrError.getConfig();
                    attrs.set(GrpcAttributes.NAME_RESOLVER_SERVICE_CONFIG, config);
                }
            } else {
                logger.log(Level.FINE, "No TXT records found for {0}", new Object[]{DnsNameResolver.this.host});
            }
            this.savedListener.onAddresses(servers, attrs.build());
        }
    }
}

