/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.ComponentModel;
using System.Collections;
using System.Globalization;

namespace PostgreSql.Data.PgSqlClient
{
	[ListBindable(false),
	Editor(typeof(Design.PgParameterCollectionEditor), typeof(System.Drawing.Design.UITypeEditor))]
	public sealed class PgParameterCollection : MarshalByRefObject, IDataParameterCollection, IList, ICollection, IEnumerable
	{	
		#region Fields

		private ArrayList parameters = new ArrayList();

		#endregion

		#region Properties

		object IDataParameterCollection.this[string parameterName] 
		{
			get { return this[parameterName]; }
			set { this[parameterName] = (PgParameter)value; }
		}

		[Browsable(false),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public PgParameter this[string parameterName]
		{
			get { return (PgParameter)this[IndexOf(parameterName)]; }
			set { this[IndexOf(parameterName)] = (PgParameter)value; }
		}

		object IList.this[int parameterIndex]
		{
			get { return (PgParameter)parameters[parameterIndex]; }
			set { parameters[parameterIndex] = (PgParameter)value; }
		}

		[Browsable(false),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public PgParameter this[int parameterIndex]
		{
			get { return (PgParameter)parameters[parameterIndex]; }
			set { parameters[parameterIndex] = (PgParameter)value; }
		}
		
		#endregion

		#region Constructors

		internal PgParameterCollection()
		{
		}

		#endregion

		#region ILIST_PROPERTIES

		bool IList.IsFixedSize
		{
			get { return parameters.IsFixedSize; }
		}

		bool IList.IsReadOnly
		{
			get { return parameters.IsReadOnly; }
		}

		#endregion

		#region ICOLLECTION_PROPERTIES

		[Browsable(false),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public int Count 
		{
			get { return parameters.Count; }
		}
		
		bool ICollection.IsSynchronized 
		{
			get { return parameters.IsSynchronized; }
		}

		object ICollection.SyncRoot 
		{
			get { return parameters.SyncRoot; }
		}

		#endregion

		#region ICOLLECTION_METHODS

		public void CopyTo(Array array, int index)
		{
			parameters.CopyTo(array, index);
		}

		#endregion

		#region ILIST_METHODS

		public void Clear()
		{
			parameters.Clear();
		}

		#endregion

		#region IENUMERABLE_METHODS

		public IEnumerator GetEnumerator()
		{
			return parameters.GetEnumerator();
		}

		#endregion

		#region Methods

		public bool Contains(object value)
		{
			return parameters.Contains(value);
		}

		public bool Contains(string parameterName)
		{
			return(-1 != IndexOf(parameterName));
		}

		public int IndexOf(object value)
		{
			return parameters.IndexOf(value);
		}

		public int IndexOf(string parameterName)
		{
			int index = 0;

			foreach (PgParameter item in this.parameters)
			{
				if (cultureAwareCompare(item.ParameterName, parameterName))
				{
					return index;
				}
				index++;
			}

			return -1;
		}

		public void Insert(int index, object value)
		{
			parameters.Insert(index, value);
		}

		public void Remove(object value)
		{
			if (!(value is PgParameter))
			{
				throw new InvalidCastException("The parameter passed was not a PgParameter.");
			}

			if (!Contains(value))
			{
				throw new SystemException("The parameter does not exist in the collection.");
			}

			parameters.Remove(value);
		}

		public void RemoveAt(int index)
		{
			this.parameters.RemoveAt(index);
		}

		public void RemoveAt(string parameterName)
		{
			RemoveAt(IndexOf(parameterName));
		}

		public int Add(object value)
		{
			if (!(value is PgParameter))
			{
				throw new InvalidCastException("The parameter passed was not a PgParameter.");
			}

			return parameters.Add((PgParameter)value);
		}

		public PgParameter Add(PgParameter param)
		{
			if (param.ParameterName != null)
			{
				parameters.Add(param);
				
				return param;
			}
			else
			{
				throw new ArgumentException("parameter must be named");
			}
		}

		public PgParameter Add(string parameterName, PgDbType PgType)
		{
			PgParameter param = new PgParameter(parameterName, PgType);			
			
			return Add(param);
		}

		public PgParameter Add(string parameterName, object value)
		{
			PgParameter param = new PgParameter(parameterName, value);

			return Add(param);
		}

		public PgParameter Add(string parameterName, PgDbType PgType, int size)
		{
			PgParameter param = new PgParameter(parameterName, PgType, size);

			return Add(param);		
		}

		public PgParameter Add(string parameterName, PgDbType PgType, int size, string sourceColumn)
		{
			PgParameter param = new PgParameter(parameterName, PgType, size, sourceColumn);

			return Add(param);		
		}

		private bool cultureAwareCompare(string strA, string strB)
		{
			return CultureInfo.CurrentCulture.CompareInfo.Compare(
				strA, 
				strB, 
				CompareOptions.IgnoreKanaType | CompareOptions.IgnoreWidth | 
				CompareOptions.IgnoreCase) == 0 ? true : false;
		}

		#endregion
	}
}