/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.internal.model;

import static org.hamcrest.CoreMatchers.*;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import static org.springframework.config.java.internal.util.AnnotationExtractionUtils.extractMethodAnnotation;

import org.junit.Test;

import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.internal.model.JavaConfigMethod.IncompatibleAnnotationError;
import org.springframework.config.java.internal.model.JavaConfigMethod.PrivateMethodError;
import org.springframework.config.java.internal.util.MethodAnnotationPrototype;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.Modifier;

import java.util.ArrayList;


/**
 * Unit tests for {@link BeanMethod}.
 *
 * @author  Chris Beams
 */
public class BeanMethodTests {

    public static final Bean FINAL_BEAN_ANNOTATION = extractMethodAnnotation(Bean.class, new MethodAnnotationPrototype() {
            @Bean(allowOverriding = false)
            public void targetMethod() { }
        }.getClass());

    static final Bean DEFAULT_METADATA = extractMethodAnnotation(Bean.class, new MethodAnnotationPrototype() {
            @Bean
            public void targetMethod() { }
        }.getClass());

    public static final BeanMethod VALID_BEAN_METHOD = new BeanMethod("m", 0, DEFAULT_METADATA);

    // ------------------------------
    // Equivalence tests
    // ------------------------------
    @Test
    public void equivalentMethodsAreEqual() {
        BeanMethod methodA = new BeanMethod("foo");
        BeanMethod methodB = new BeanMethod("foo");

        assertThat(methodA, equalTo(methodB));
    }

    @Test
    public void methodsWithDifferentModifiersAreNotEqual() {
        BeanMethod methodA = new BeanMethod("foo");
        BeanMethod methodB = new BeanMethod("foo", Modifier.PUBLIC);

        assertThat(methodA, not(equalTo(methodB)));
    }

    /*
     * creating a new BeanMethod("foo") is equivalent to a class that declares:
     *
     *     class LineNumberConfig {
     *         @Bean TestBean foo() { ... }
     *     }
     */
    @Test
    public void byDefaultMethodShouldHaveNoModifiers() {
        BeanMethod method = new BeanMethod("foo");

        int modifiers = method.getModifiers();

        // 0 signifies 'no modifiers' - see java.lang.reflect.Modifier
        assertEquals(0, modifiers);
    }

    // ------------------------------
    // Validation tests
    // ------------------------------

    @Test
    public void privateBeanMethodsAreNotValid() {
        ArrayList<UsageError> errors = new ArrayList<UsageError>();
        new BeanMethod("foo", Modifier.PRIVATE).detectUsageErrors(errors);
        assertThat(errors.get(0), instanceOf(PrivateMethodError.class));
    }

    @Test
    public void unknownAnnotationCausesIncompatibilityError() {
        Bean beanAnno = DEFAULT_METADATA;
        Bogus unknownAnno = extractMethodAnnotation(Bogus.class, new MethodAnnotationPrototype() {
                @Bogus
                public void targetMethod() { }
            }.getClass());

        ArrayList<UsageError> errors = new ArrayList<UsageError>();

        BeanMethod method = new BeanMethod("m", 0, beanAnno, unknownAnno);
        method.detectUsageErrors(errors);
        assertThat(errors.get(0), instanceOf(IncompatibleAnnotationError.class));
    }


    @Target(ElementType.METHOD)
    @Retention(RetentionPolicy.RUNTIME)
    static @interface Bogus { }
}
