// Command protoc-gen-grpchan is a protoc plugin that generates gRPC client stubs
// in Go that use github.com/fullstorydev/grpchan.Channel as their transport
// abstraction, instead of using *grpc.ClientConn. This can be used to carry RPC
// requests and streams over other transports, such as HTTP 1.1 or in-process.
package main

import (
	"fmt"
	"path"
	"strings"
	"text/template"

	"github.com/jhump/gopoet"
	"github.com/jhump/goprotoc/plugins"
	"github.com/jhump/protoreflect/desc"
	"google.golang.org/protobuf/types/pluginpb"
)

func main() {
	plugins.PluginMain(doCodeGen)
}

func doCodeGen(req *plugins.CodeGenRequest, resp *plugins.CodeGenResponse) error {
	resp.SupportsFeatures(pluginpb.CodeGeneratorResponse_FEATURE_PROTO3_OPTIONAL)
	args, err := parseArgs(req.Args)
	if err != nil {
		return err
	}
	names := plugins.GoNames{
		ImportMap:      args.importMap,
		ModuleRoot:     args.moduleRoot,
		SourceRelative: args.sourceRelative,
	}
	if args.importPath != "" {
		// if we're overriding import path, go ahead and query
		// package for each file, which will cache the override name
		// so all subsequent queries are consistent
		for _, fd := range req.Files {
			// Only use the override for files that don't otherwise have an
			// entry in the specified import map
			if _, ok := args.importMap[fd.GetName()]; !ok {
				names.GoPackageForFileWithOverride(fd, args.importPath)
			}
		}
	}
	for _, fd := range req.Files {
		if err := generateChanStubs(fd, &names, resp, args); err != nil {
			if fe, ok := err.(*gopoet.FormatError); ok {
				if args.debug {
					return fmt.Errorf("%s: error in generated Go code: %v:\n%s", fd.GetName(), err, fe.Unformatted)
				} else {
					return fmt.Errorf("%s: error in generated Go code: %v (use debug=true arg to show full source)", fd.GetName(), err)
				}
			} else {
				return fmt.Errorf("%s: %v", fd.GetName(), err)
			}
		}
	}
	return nil
}

var typeOfRegistry = gopoet.NamedType(gopoet.NewSymbol("github.com/fullstorydev/grpchan", "ServiceRegistry"))
var typeOfClientConn = gopoet.NamedType(gopoet.NewSymbol("google.golang.org/grpc", "ClientConnInterface"))
var typeOfContext = gopoet.NamedType(gopoet.NewSymbol("context", "Context"))
var typeOfCallOptions = gopoet.SliceType(gopoet.NamedType(gopoet.NewSymbol("google.golang.org/grpc", "CallOption")))

func generateChanStubs(fd *desc.FileDescriptor, names *plugins.GoNames, resp *plugins.CodeGenResponse, args codeGenArgs) error {
	if len(fd.GetServices()) == 0 {
		return nil
	}

	pkg := names.GoPackageForFile(fd)
	filename := names.OutputFilenameFor(fd, ".pb.grpchan.go")
	f := gopoet.NewGoFile(path.Base(filename), pkg.ImportPath, pkg.Name)

	f.FileComment = "Code generated by protoc-gen-grpchan. DO NOT EDIT.\n" +
		"source: " + fd.GetName()

	for _, sd := range fd.GetServices() {
		svcName := names.CamelCase(sd.GetName())
		lowerSvcName := gopoet.Unexport(svcName)

		f.AddElement(gopoet.NewFunc(fmt.Sprintf("RegisterHandler%s", svcName)).
			SetComment(fmt.Sprintf("Deprecated: Use Register%sServer instead.", svcName)).
			AddArg("reg", typeOfRegistry).
			AddArg("srv", names.GoTypeForServiceServer(sd)).
			Printlnf("reg.RegisterService(&%s, srv)", serviceDescVarName(sd, names, args.legacyDescNames)))

		if !args.legacyStubs {
			continue
		}

		cc := gopoet.NewStructTypeSpec(fmt.Sprintf("%sChannelClient", lowerSvcName),
			gopoet.NewField("ch", typeOfClientConn))
		f.AddType(cc)

		f.AddElement(gopoet.NewFunc(fmt.Sprintf("New%sChannelClient", svcName)).
			SetComment(fmt.Sprintf("Deprecated: Use New%sClient instead.", svcName)).
			AddArg("ch", typeOfClientConn).
			AddResult("", names.GoTypeForServiceClient(sd)).
			SetComment(fmt.Sprintf("Deprecated: Use New%sClient instead.", svcName)).
			Printlnf("return &%s{ch: ch}", cc))

		streamCount := 0
		tmpls := templates{}
		for _, md := range sd.GetMethods() {
			methodInfo := struct {
				ServiceName  string
				MethodName   string
				ServiceDesc  string
				StreamClient string
				StreamIndex  int
				RequestType  gopoet.TypeName
			}{
				ServiceName:  sd.GetFullyQualifiedName(),
				MethodName:   md.GetName(),
				ServiceDesc:  serviceDescVarName(sd, names, args.legacyDescNames),
				StreamClient: names.GoTypeForStreamClientImpl(md),
				StreamIndex:  streamCount,
				RequestType:  names.GoTypeForMessage(md.GetOutputType()),
			}
			mtdName := names.CamelCase(md.GetName())
			if md.IsClientStreaming() {
				// bidi or client streaming method
				f.AddElement(gopoet.NewMethod(gopoet.NewPointerReceiverForType("c", cc), mtdName).
					AddArg("ctx", typeOfContext).
					AddArg("opts", typeOfCallOptions).
					SetVariadic(true).
					AddResult("", names.GoTypeForStreamClient(md)).
					AddResult("", gopoet.ErrorType).
					RenderCode(tmpls.makeTemplate(
						`stream, err := c.ch.NewStream(ctx, &{{.ServiceDesc}}.Streams[{{.StreamIndex}}], "/{{.ServiceName}}/{{.MethodName}}", opts...)
						if err != nil {
							return nil, err
						}
						x := &{{.StreamClient}}{stream}
						return x, nil`), &methodInfo))
				streamCount++
			} else if md.IsServerStreaming() {
				// server streaming method
				f.AddElement(gopoet.NewMethod(gopoet.NewPointerReceiverForType("c", cc), mtdName).
					AddArg("ctx", typeOfContext).
					AddArg("in", names.GoTypeOfRequest(md)).
					AddArg("opts", typeOfCallOptions).
					SetVariadic(true).
					AddResult("", names.GoTypeForStreamClient(md)).
					AddResult("", gopoet.ErrorType).
					RenderCode(tmpls.makeTemplate(
						`stream, err := c.ch.NewStream(ctx, &{{.ServiceDesc}}.Streams[{{.StreamIndex}}], "/{{.ServiceName}}/{{.MethodName}}", opts...)
						if err != nil {
							return nil, err
						}
						x := &{{.StreamClient}}{stream}
						if err := x.ClientStream.SendMsg(in); err != nil {
						    return nil, err
						}
						if err := x.ClientStream.CloseSend(); err != nil {
						    return nil, err
						}
						return x, nil`), &methodInfo))
				streamCount++
			} else {
				// unary method
				f.AddElement(gopoet.NewMethod(gopoet.NewPointerReceiverForType("c", cc), mtdName).
					AddArg("ctx", typeOfContext).
					AddArg("in", names.GoTypeOfRequest(md)).
					AddArg("opts", typeOfCallOptions).
					SetVariadic(true).
					AddResult("", names.GoTypeOfResponse(md)).
					AddResult("", gopoet.ErrorType).
					RenderCode(tmpls.makeTemplate(
						`out := new({{.RequestType}})
						err := c.ch.Invoke(ctx, "/{{.ServiceName}}/{{.MethodName}}", in, out, opts...)
						if err != nil {
							return nil, err
						}
						return out, nil`), &methodInfo))
			}
		}
	}

	out := resp.OutputFile(filename)
	return gopoet.WriteGoFile(out, f)
}

func serviceDescVarName(sd *desc.ServiceDescriptor, names *plugins.GoNames, legacyNames bool) string {
	if legacyNames {
		return names.GoNameOfServiceDesc(sd)
	}
	return names.GoNameOfExportedServiceDesc(sd).Name
}

type templates map[string]*template.Template

func (t templates) makeTemplate(templateText string) *template.Template {
	tpl := t[templateText]
	if tpl == nil {
		tpl = template.Must(template.New("code").Parse(templateText))
		t[templateText] = tpl
	}
	return tpl
}

type codeGenArgs struct {
	debug           bool
	legacyStubs     bool
	legacyDescNames bool
	importPath      string
	importMap       map[string]string
	moduleRoot      string
	sourceRelative  bool
}

func parseArgs(args []string) (codeGenArgs, error) {
	var result codeGenArgs
	for _, arg := range args {
		vals := strings.SplitN(arg, "=", 2)
		switch vals[0] {
		case "debug":
			val, err := boolVal(vals)
			if err != nil {
				return result, err
			}
			result.debug = val

		case "legacy_stubs":
			val, err := boolVal(vals)
			if err != nil {
				return result, err
			}
			result.legacyStubs = val

		case "legacy_desc_names":
			val, err := boolVal(vals)
			if err != nil {
				return result, err
			}
			result.legacyDescNames = val

		case "import_path":
			if len(vals) == 1 {
				return result, fmt.Errorf("plugin option 'import_path' requires an argument")
			}
			result.importPath = vals[1]

		case "module":
			if len(vals) == 1 {
				return result, fmt.Errorf("plugin option 'module' requires an argument")
			}
			result.moduleRoot = vals[1]

		case "paths":
			if len(vals) == 1 {
				return result, fmt.Errorf("plugin option 'paths' requires an argument")
			}
			switch vals[1] {
			case "import":
				result.sourceRelative = false
			case "source_relative":
				result.sourceRelative = true
			default:
				return result, fmt.Errorf("plugin option 'paths' accepts 'import' or 'source_relative' as value, got %q", vals[1])
			}

		default:
			if len(vals[0]) > 1 && vals[0][0] == 'M' {
				if len(vals) == 1 {
					return result, fmt.Errorf("plugin 'M' options require an argument: %s", vals[0])
				}
				if result.importMap == nil {
					result.importMap = map[string]string{}
				}
				result.importMap[vals[0][1:]] = vals[1]
				break
			}

			return result, fmt.Errorf("unknown plugin option: %s", vals[0])
		}
	}

	if result.sourceRelative && result.moduleRoot != "" {
		return result, fmt.Errorf("plugin option 'module' cannot be used with 'paths=source_relative'")
	}

	return result, nil
}

func boolVal(vals []string) (bool, error) {
	if len(vals) == 1 {
		// if no value, assume "true"
		return true, nil
	}
	switch strings.ToLower(vals[1]) {
	case "true", "on", "yes", "1":
		return true, nil
	case "false", "off", "no", "0":
		return false, nil
	default:
		return false, fmt.Errorf("invalid boolean arg for option '%s': %s", vals[0], vals[1])
	}
}
