// Copyright ©2015 The Gonum Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package testlapack

import (
	"testing"

	"gonum.org/v1/gonum/floats"
)

type Dlasq5er interface {
	Dlasq5(i0, n0 int, z []float64, pp int, tau, sigma float64) (i0Out, n0Out, ppOut int, tauOut, sigmaOut, dmin, dmin1, dmin2, dn, dnm1, dnm2 float64)
}

func Dlasq5Test(t *testing.T, impl Dlasq5er) {
	dTol := 1e-14
	for _, test := range []struct {
		z          []float64
		i0, n0, pp int
		tau, sigma float64

		i0Out, n0Out, ppOut         int
		tauOut, sigmaOut            float64
		dminOut, dmin1Out, dmin2Out float64
		dnOut, dnm1Out, dnm2Out     float64
	}{
		{
			z:        []float64{0.1914365246180821e+01, 0.1564384297703890e+01, 0.2493389162143899e+00, 0.3499809484769305e+00, 0.1315996513131545e+01, 0.1363862112490627e+01, 0.9898466611970759e-01, 0.2014733168553078e+00, 0.6023973979587287e+00, 0.6465544792741794e+00, 0.2210033410638781e-02, 0.5482758480425683e-01, 0.9861857233678967e-01, 0.2428190810745492e-01, 0.4756321484454819e+00, 0.7654669763997353e-01, 0.2588748143677115e+00, 0.6127784069508770e+00, 0.1078611376690004e+00, 0.1217285558623164e+00, 0.6442896492255246e+00, 0.2293835804898155e+00, 0.6203230486639705e+00, 0.5227672064047094e+00, 0.3695660678607585e+00, 0.7645233184745865e+00, 0.5378838054252265e+00, 0.2253657980501426e+00, 0.3562533181264623e+00, 0.8820486722335483e+00, 0.2222132496436145e-01, 0.1208845131814035e-01, 0.1275094303021685e+01, 0.6548746852163357e+00, 0.1647324354821218e+00, 0.6424409427697111e+00, 0.1007530576543866e+01, 0.3269551736546701e+00, 0.3453881601783118e+00, 0.8453078383713172e+00, 0.2679391719153404e+00, 0.4116714838778281e+00, 0.7328677736683723e+00, 0.2016558482158241e+00, 0.8360828138307410e+00, 0.9737579452195326e+00, 0.4813660709592822e+00, 0.5951926422795808e+00, 0.6495370513676459e+00, 0.6761876248148171e+00, 0.2325475880222648e+00, 0.4547154975121112e+00, 0.1993624802893807e+00, 0.3321819367342255e+00, 0.3782318916911257e+00, 0.9972813157741996e-01, 0.9830449403503746e+00, 0.7561080996844842e+00, 0.4429733864040367e+00, 0.6051687323570161e+00, 0.1173279550602403e+01, 0.7195724480316686e+00, 0.5035524069144587e+00, 0.8966804889747714e+00, 0.3058980395058521e+00, 0.6588832353928662e+00, 0.3014634433415453e+00, 0.1505672110274446e+00, 0.1289422237567578e+01, 0.6124645310993601e+00, 0.7583364305799440e+00, 0.9784211498097629e+00, 0.4977814779461571e+00, 0.9993813577491869e+00, 0.2841468847862598e+00, 0.2567365507769143e+00, 0.9257539794205765e+00, 0.5509268385614666e+00, 0.5231355605450990e-04, 0.6589740256453697e+00, 0.2117869221381033e-04, 0.7349224826832024e-04, 0.0000000000000000e+00, 0.0000000000000000e+00},
			i0:       1,
			n0:       21,
			pp:       0,
			tau:      0.0000000000000000,
			sigma:    0.0000000000000000,
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			dminOut:  2.1175426017541180e-005,
			dmin1Out: 4.4311601260836921e-002,
			dmin2Out: 4.4311601260836921e-002,
			dnOut:    2.1175426017541180e-005,
			dnm1Out:  0.33915960483100382,
			dnm2Out:  0.16428924199195991,
		},
		{
			z:        []float64{0.1914365246180821e+01, 0.2163704162395211e+01, 0.2493389162143899e+00, 0.1516515751224039e+00, 0.1315996513131545e+01, 0.1263329604128848e+01, 0.9898466611970759e-01, 0.4719916727467415e-01, 0.6023973979587287e+00, 0.5574082640946934e+00, 0.2210033410638781e-02, 0.3910066531356214e-03, 0.9861857233678967e-01, 0.5738597141291359e+00, 0.4756321484454819e+00, 0.2145632131068746e+00, 0.2588748143677115e+00, 0.1521727389298373e+00, 0.1078611376690004e+00, 0.4566771620366771e+00, 0.6442896492255246e+00, 0.8079355358528180e+00, 0.6203230486639705e+00, 0.2837483186776231e+00, 0.3695660678607585e+00, 0.6237015546083620e+00, 0.5378838054252265e+00, 0.3072349091217998e+00, 0.3562533181264623e+00, 0.7123973396902394e-01, 0.2222132496436145e-01, 0.3977314805803597e+00, 0.1275094303021685e+01, 0.1042095257923447e+01, 0.1647324354821218e+00, 0.1592685164190333e+00, 0.1007530576543866e+01, 0.1193650220303144e+01, 0.3453881601783118e+00, 0.7752942700755104e-01, 0.2679391719153404e+00, 0.9232775185761617e+00, 0.7328677736683723e+00, 0.6636554427529671e+00, 0.8360828138307410e+00, 0.6537934420370561e+00, 0.4813660709592822e+00, 0.4782322339990674e+00, 0.6495370513676459e+00, 0.4038524053908432e+00, 0.2325475880222648e+00, 0.1147975431483785e+00, 0.1993624802893807e+00, 0.4627968288321279e+00, 0.3782318916911257e+00, 0.8034172324482011e+00, 0.9830449403503746e+00, 0.6226010943062101e+00, 0.4429733864040367e+00, 0.8347746582554776e+00, 0.1173279550602403e+01, 0.8420572992613844e+00, 0.5035524069144587e+00, 0.1829278057427913e+00, 0.3058980395058521e+00, 0.4244336771046062e+00, 0.3014634433415453e+00, 0.9158407747236312e+00, 0.1289422237567578e+01, 0.1131917893423890e+01, 0.7583364305799440e+00, 0.3334922359541972e+00, 0.4977814779461571e+00, 0.4484361267782198e+00, 0.2841468847862598e+00, 0.5865943745895725e+00, 0.9257539794205765e+00, 0.3392119183870583e+00, 0.5231355605450990e-04, 0.3266196269153995e-08, 0.2117869221381033e-04, 0.2117542601754118e-04, 0.0000000000000000e+00, 0.3910066531356214e-03, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			tau:      0.0000000000000000,
			sigma:    0.0000000000000000,
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			dminOut:  2.1175425069176302e-005,
			dmin1Out: 2.9944624525135358e-002,
			dmin2Out: 2.9944624525135358e-002,
			dnOut:    2.1175425069176302e-005,
			dnm1Out:  7.2928780948497918e-002,
			dnm2Out:  0.16065460645225654,
		},
		{
			z:        []float64{0.2315355737517615e+01, 0.2163704162395211e+01, 0.8274578340618610e-01, 0.1516515751224039e+00, 0.1227782987997336e+01, 0.1263329604128848e+01, 0.2142822156235013e-01, 0.4719916727467415e-01, 0.5363710491854788e+00, 0.5574082640946934e+00, 0.4183353417969536e-03, 0.3910066531356214e-03, 0.7880045918942136e+00, 0.5738597141291359e+00, 0.4143462125464707e-01, 0.2145632131068746e+00, 0.5674152797118673e+00, 0.1521727389298373e+00, 0.6502569120260687e+00, 0.4566771620366771e+00, 0.4414269425043723e+00, 0.8079355358528180e+00, 0.4009140594652070e+00, 0.2837483186776231e+00, 0.5300224042649548e+00, 0.6237015546083620e+00, 0.4129510944388858e-01, 0.3072349091217998e+00, 0.4276761051054951e+00, 0.7123973396902394e-01, 0.9691308092544145e+00, 0.3977314805803597e+00, 0.2322329650880660e+00, 0.1042095257923447e+01, 0.8186215063776209e+00, 0.1592685164190333e+00, 0.4525581409330741e+00, 0.1193650220303144e+01, 0.1581701233715052e+00, 0.7752942700755104e-01, 0.1428762837957623e+01, 0.9232775185761617e+00, 0.3036848136842134e+00, 0.6636554427529671e+00, 0.8283408623519102e+00, 0.6537934420370561e+00, 0.2331591338951825e+00, 0.4782322339990674e+00, 0.2854908146440392e+00, 0.4038524053908432e+00, 0.1860933389154074e+00, 0.1147975431483785e+00, 0.1080120722364922e+01, 0.4627968288321279e+00, 0.4631042046962229e+00, 0.8034172324482011e+00, 0.9942715478654648e+00, 0.6226010943062101e+00, 0.7069779837626068e+00, 0.8347746582554776e+00, 0.3180071212415688e+00, 0.8420572992613844e+00, 0.2441477440283845e+00, 0.1829278057427913e+00, 0.1096126707799853e+01, 0.4244336771046062e+00, 0.9457451890006905e+00, 0.9158407747236312e+00, 0.5196649403773971e+00, 0.1131917893423890e+01, 0.2877815203259632e+00, 0.3334922359541972e+00, 0.7472489810418290e+00, 0.4484361267782198e+00, 0.2662831374385604e+00, 0.5865943745895725e+00, 0.7292878421469419e-01, 0.3392119183870583e+00, 0.9483648767903632e-12, 0.3266196269153995e-08, 0.2117542506917630e-04, 0.2117542601754118e-04, 0.4183353417969536e-03, 0.3910066531356214e-03},
			i0:       1,
			n0:       21,
			pp:       0,
			tau:      2.1175313795360271e-005,
			sigma:    0.0000000000000000,
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			tauOut:   2.1175313795360271e-005,
			sigmaOut: 0.0000000000000000,
			dminOut:  1.1127325659669794e-010,
			dmin1Out: 3.1433071595911154e-002,
			dmin2Out: 3.1433071595911154e-002,
			dnOut:    1.1127325659669794e-010,
			dnm1Out:  3.5896964560873705e-002,
			dnm2Out:  0.25842281720128102,
		},
		{
			z:        []float64{0.2315355737517615e+01, 0.2398080345610006e+01, 0.8274578340618610e-01, 0.4236466279397526e-01, 0.1227782987997336e+01, 0.1206825371451915e+01, 0.2142822156235013e-01, 0.9523728911788614e-02, 0.5363710491854788e+00, 0.5272444803016919e+00, 0.4183353417969536e-03, 0.6252320936560726e-03, 0.7880045918942136e+00, 0.8287928057414093e+00, 0.4143462125464707e-01, 0.2836732781232222e-01, 0.5674152797118673e+00, 0.1189283688611819e+01, 0.6502569120260687e+00, 0.2413561400585997e+00, 0.4414269425043723e+00, 0.6009636865971842e+00, 0.4009140594652070e+00, 0.3535878097802652e+00, 0.5300224042649548e+00, 0.2177085286147829e+00, 0.4129510944388858e-01, 0.8112190955144877e-01, 0.4276761051054951e+00, 0.1315663829494665e+01, 0.9691308092544145e+00, 0.1710650671895379e+00, 0.2322329650880660e+00, 0.8797682289623537e+00, 0.8186215063776209e+00, 0.4211038940233675e+00, 0.4525581409330741e+00, 0.1896031949674164e+00, 0.1581701233715052e+00, 0.1191897606932286e+01, 0.1428762837957623e+01, 0.5405288693957555e+00, 0.3036848136842134e+00, 0.4653859482687157e+00, 0.8283408623519102e+00, 0.5960928726645816e+00, 0.2331591338951825e+00, 0.1116684901463164e+00, 0.2854908146440392e+00, 0.3598944880993349e+00, 0.1860933389154074e+00, 0.5585061130503639e+00, 0.1080120722364922e+01, 0.9846976386969850e+00, 0.4631042046962229e+00, 0.4676068229793028e+00, 0.9942715478654648e+00, 0.1233621533334973e+01, 0.7069779837626068e+00, 0.1822471700779458e+00, 0.3180071212415688e+00, 0.3798865198782122e+00, 0.2441477440283845e+00, 0.7044652781161848e+00, 0.1096126707799853e+01, 0.1337385443370563e+01, 0.9457451890006905e+00, 0.3674861422265960e+00, 0.5196649403773971e+00, 0.4399391431629689e+00, 0.2877815203259632e+00, 0.4888049885267526e+00, 0.7472489810418290e+00, 0.5247059546398414e+00, 0.2662831374385604e+00, 0.3701064434002514e-01, 0.7292878421469419e-01, 0.3589696456182207e-01, 0.9483648767903632e-12, 0.5594353069081231e-15, 0.2117542506917630e-04, 0.1112732565966979e-09, 0.4183353417969536e-03, 0.6252320936560726e-03, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			tau:      2.1175313795360271e-005,
			sigma:    2.1175313795360271e-005,
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			tauOut:   2.1175313795360271e-005,
			sigmaOut: 2.1175313795360271e-005,
			dminOut:  -2.1175202522103674e-005,
			dmin1Out: 2.9116497146097618e-002,
			dmin2Out: 4.9396687496051764e-002,
			dnOut:    -2.1175202522103674e-005,
			dnm1Out:  2.9116497146097618e-002,
			dnm2Out:  0.15954393093937583,
		},
		{
			z:        []float64{0.2440423833090186e+01, 0.2398080345610006e+01, 0.2094994698033050e-01, 0.4236466279397526e-01, 0.1195377978069578e+01, 0.1206825371451915e+01, 0.4200624064314086e-02, 0.9523728911788614e-02, 0.5236479130172386e+00, 0.5272444803016919e+00, 0.9895730475750664e-03, 0.6252320936560726e-03, 0.8561493851923613e+00, 0.8287928057414093e+00, 0.3940527300515336e-01, 0.2836732781232222e-01, 0.1391213380351470e+01, 0.1189283688611819e+01, 0.1042588274099539e+00, 0.2413561400585997e+00, 0.8502714936537001e+00, 0.6009636865971842e+00, 0.9053470847599347e-01, 0.3535878097802652e+00, 0.2082745543764428e+00, 0.2177085286147829e+00, 0.5124445590385125e+00, 0.8112190955144877e-01, 0.9742631623318954e+00, 0.1315663829494665e+01, 0.1544732645319877e+00, 0.1710650671895379e+00, 0.1146377683139938e+01, 0.8797682289623537e+00, 0.6964776521238716e-01, 0.4211038940233675e+00, 0.1311831861373520e+01, 0.1896031949674164e+00, 0.4911110065859084e+00, 0.1191897606932286e+01, 0.5147826357647675e+00, 0.5405288693957555e+00, 0.5388939477127089e+00, 0.4653859482687157e+00, 0.1688462397843937e+00, 0.5960928726645816e+00, 0.2380205454936569e+00, 0.1116684901463164e+00, 0.6803588803422466e+00, 0.3598944880993349e+00, 0.8083375797812383e+00, 0.5585061130503639e+00, 0.6439457065812542e+00, 0.9846976386969850e+00, 0.8958050967125053e+00, 0.4676068229793028e+00, 0.5200424313866183e+00, 0.1233621533334973e+01, 0.1331299890548604e+00, 0.1822471700779458e+00, 0.9512006336257413e+00, 0.3798865198782122e+00, 0.9904762202705547e+00, 0.7044652781161848e+00, 0.7143741900128092e+00, 0.1337385443370563e+01, 0.2263121215682984e+00, 0.3674861422265960e+00, 0.7024108348076278e+00, 0.4399391431629689e+00, 0.3651408483866702e+00, 0.4888049885267526e+00, 0.1965545752794010e+00, 0.5247059546398414e+00, 0.6759292101929097e-02, 0.3701064434002514e-01, 0.2911649714609818e-01, 0.3589696456182207e-01, 0.2137969692662087e-23, 0.5594353069081231e-15, -0.2117520252210367e-04, 0.1112732565966979e-09, 0.9895730475750664e-03, 0.6252320936560726e-03, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			tau:      1.1127325659669789e-010,
			sigma:    2.1175313795360271e-005,
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			tauOut:   1.1127325659669789e-010,
			sigmaOut: 2.1175313795360271e-005,
			dminOut:  -2.0808762284537102e-024,
			dmin1Out: 2.9139336744737766e-002,
			dmin2Out: 4.9426557292086552e-002,
			dnOut:    -2.0808762284537102e-024,
			dnm1Out:  2.9139336744737766e-002,
			dnm2Out:  0.15959234211062134,
		},
		{
			z:        []float64{0.2440445008292708e+01, 0.2398080345610006e+01, 0.2094976520226600e-01, 0.4236466279397526e-01, 0.1195399335050165e+01, 0.1206825371451915e+01, 0.4200549016048655e-02, 0.9523728911788614e-02, 0.5236691632680260e+00, 0.5272444803016919e+00, 0.9895328911616120e-03, 0.6252320936560726e-03, 0.8561706005512968e+00, 0.8287928057414093e+00, 0.3940429656773515e-01, 0.2836732781232222e-01, 0.1391235531991410e+01, 0.1189283688611819e+01, 0.1042571673718422e+00, 0.2413561400585997e+00, 0.8502943288943339e+00, 0.6009636865971842e+00, 0.9053227710395735e-01, 0.3535878097802652e+00, 0.2082981609510011e+00, 0.2177085286147829e+00, 0.5123864833424303e+00, 0.8112190955144877e-01, 0.9743424132304999e+00, 0.1315663829494665e+01, 0.1544607000116935e+00, 0.1710650671895379e+00, 0.1146411422862754e+01, 0.8797682289623537e+00, 0.6964571542795012e-01, 0.4211038940233675e+00, 0.1311855086360479e+01, 0.1896031949674164e+00, 0.4911023119923957e+00, 0.1191897606932286e+01, 0.5148125055608023e+00, 0.5405288693957555e+00, 0.5388626806938843e+00, 0.4653859482687157e+00, 0.1688986820057405e+00, 0.5960928726645816e+00, 0.2379466412690434e+00, 0.1116684901463164e+00, 0.6804539597693821e+00, 0.3598944880993349e+00, 0.8082246312519304e+00, 0.5585061130503639e+00, 0.6440798303130841e+00, 0.9846976386969850e+00, 0.8956185534970393e+00, 0.4676068229793028e+00, 0.5202501498046066e+00, 0.1233621533334973e+01, 0.1330768347199243e+00, 0.1822471700779458e+00, 0.9512749631631994e+00, 0.3798865198782122e+00, 0.9903988276741268e+00, 0.7044652781161848e+00, 0.7144727578117591e+00, 0.1337385443370563e+01, 0.2262808998212762e+00, 0.3674861422265960e+00, 0.7024632317571722e+00, 0.4399391431629689e+00, 0.3651136124179467e+00, 0.4888049885267526e+00, 0.1966029864506465e+00, 0.5247059546398414e+00, 0.6757627705811050e-02, 0.3701064434002514e-01, 0.2913933674473832e-01, 0.3589696456182207e-01, 0.2136293938333395e-23, 0.5594353069081231e-15, 0.0000000000000000e+00, 0.1112732565966979e-09, 0.9895328911616120e-03, 0.6252320936560726e-03},
			i0:       1,
			n0:       21,
			pp:       0,
			tau:      -0.0000000000000000,
			sigma:    2.1175425068616867e-005,
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 2.1175425068616867e-005,
			dminOut:  0.0000000000000000,
			dmin1Out: 2.7016889331018056e-002,
			dmin2Out: 5.3061698118516694e-002,
			dnOut:    0.0000000000000000,
			dnm1Out:  2.7016889331018056e-002,
			dnm2Out:  8.6018658784156071e-002,
		},
		{
			z:        []float64{0.2440445008292708e+01, 0.2461394773494974e+01, 0.2094976520226600e-01, 0.1017444891892999e-01, 0.1195399335050165e+01, 0.1189425435147283e+01, 0.4200549016048655e-02, 0.1849378635683999e-02, 0.5236691632680260e+00, 0.5228093175235037e+00, 0.9895328911616120e-03, 0.1620493249248586e-02, 0.8561706005512968e+00, 0.8939544038697832e+00, 0.3940429656773515e-01, 0.6132377362967349e-01, 0.1391235531991410e+01, 0.1434168925733579e+01, 0.1042571673718422e+00, 0.6181229879703373e-01, 0.8502943288943339e+00, 0.8790143072012576e+00, 0.9053227710395735e-01, 0.2145324219750511e-01, 0.2082981609510011e+00, 0.6992314020959263e+00, 0.5123864833424303e+00, 0.7139837844669097e+00, 0.9743424132304999e+00, 0.4148193287752837e+00, 0.1544607000116935e+00, 0.4268738185358478e+00, 0.1146411422862754e+01, 0.7891833197548568e+00, 0.6964571542795012e-01, 0.1157716892137957e+00, 0.1311855086360479e+01, 0.1687185709139079e+01, 0.4911023119923957e+00, 0.1498504938454686e+00, 0.5148125055608023e+00, 0.9038246924092180e+00, 0.5388626806938843e+00, 0.1006978425303630e+00, 0.1688986820057405e+00, 0.3061474807444209e+00, 0.2379466412690434e+00, 0.5288684194677825e+00, 0.6804539597693821e+00, 0.9598101715535300e+00, 0.8082246312519304e+00, 0.5423584775195998e+00, 0.6440798303130841e+00, 0.9973399062905237e+00, 0.8956185534970393e+00, 0.4671884516860899e+00, 0.5202501498046066e+00, 0.1861385328384410e+00, 0.1330768347199243e+00, 0.6800991665489665e+00, 0.9512749631631994e+00, 0.1261574624288360e+01, 0.9903988276741268e+00, 0.5608966509936130e+00, 0.7144727578117591e+00, 0.3798570066394223e+00, 0.2262808998212762e+00, 0.4184574968871406e+00, 0.7024632317571722e+00, 0.6491193472879784e+00, 0.3651136124179467e+00, 0.1105843276664904e+00, 0.1966029864506465e+00, 0.9277628648996712e-01, 0.6757627705811050e-02, 0.2122447413720272e-02, 0.2913933674473832e-01, 0.2701688933101806e-01, 0.2136293938333395e-23, 0.0000000000000000e+00, 0.2117542506861687e-04},
			i0:       1,
			n0:       20,
			pp:       1,
			tau:      -0.0000000000000000,
			sigma:    2.1175425068616867e-005,
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 2.1175425068616867e-005,
			dminOut:  2.5763383633962696e-002,
			dmin1Out: 4.3622798915905092e-002,
			dmin2Out: 7.4536672467372611e-002,
			dnOut:    2.5763383633962696e-002,
			dnm1Out:  4.3622798915905092e-002,
			dnm2Out:  9.8141518071882677e-002,
		},
		{
			z:        []float64{0.2471569222413904e+01, 0.2461394773494974e+01, 0.4896382518051712e-02, 0.1017444891892999e-01, 0.1186378431264915e+01, 0.1189425435147283e+01, 0.8149780515932184e-03, 0.1849378635683999e-02, 0.5236148327211592e+00, 0.5228093175235037e+00, 0.2766627272719901e-02, 0.1620493249248586e-02, 0.9525115502267366e+00, 0.8939544038697832e+00, 0.9233342160256496e-01, 0.6132377362967349e-01, 0.1403647802928048e+01, 0.1434168925733579e+01, 0.3870906568602875e-01, 0.6181229879703373e-01, 0.8617584837127339e+00, 0.8790143072012576e+00, 0.1740717486950262e-01, 0.2145324219750511e-01, 0.1395808011693333e+01, 0.6992314020959263e+00, 0.2121884039551361e+00, 0.7139837844669097e+00, 0.6295047433559955e+00, 0.4148193287752837e+00, 0.5351535485381410e+00, 0.4268738185358478e+00, 0.3698014604305115e+00, 0.7891833197548568e+00, 0.5281978587564573e+00, 0.1157716892137957e+00, 0.1308838344228090e+01, 0.1687185709139079e+01, 0.1034799882693896e+00, 0.1498504938454686e+00, 0.9010425466701916e+00, 0.9038246924092180e+00, 0.3421413441684364e-01, 0.1006978425303630e+00, 0.8008017657953598e+00, 0.3061474807444209e+00, 0.6338813300623194e+00, 0.5288684194677825e+00, 0.8682873190108105e+00, 0.9598101715535300e+00, 0.6229686202966810e+00, 0.5423584775195998e+00, 0.8415597376799326e+00, 0.9973399062905237e+00, 0.1033340463692495e+00, 0.4671884516860899e+00, 0.7629036530181579e+00, 0.1861385328384410e+00, 0.1124645093942705e+01, 0.6800991665489665e+00, 0.6978261813392677e+00, 0.1261574624288360e+01, 0.3053203341720497e+00, 0.5608966509936130e+00, 0.4929941693545132e+00, 0.3798570066394223e+00, 0.5509778292160957e+00, 0.4184574968871406e+00, 0.2087258457383731e+00, 0.6491193472879784e+00, 0.4915348757406203e-01, 0.1105843276664904e+00, 0.4574524632962537e-01, 0.9277628648996712e-01, 0.1253505697055357e-02, 0.2122447413720272e-02, 0.2576338363396270e-01, 0.2701688933101806e-01, 0.8149780515932184e-03, 0.1620493249248586e-02},
			i0:       1,
			n0:       20,
			pp:       0,
			tau:      2.0080554394878082e-002,
			sigma:    2.1175425068616867e-005,
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			tauOut:   2.0080554394878082e-002,
			sigmaOut: 2.1175425068616867e-005,
			dminOut:  -2.1187919252575148e-003,
			dmin1Out: -2.1187919252575148e-003,
			dmin2Out: 3.1777270007152948e-002,
			dnOut:    4.3005217031728403e-002,
			dnm1Out:  -2.1187919252575148e-003,
			dnm2Out:  3.1777270007152948e-002,
		},
		{
			z:        []float64{0.2471569222413904e+01, 0.2456385050537078e+01, 0.4896382518051712e-02, 0.2364842030515144e-02, 0.1186378431264915e+01, 0.1164748012891115e+01, 0.8149780515932184e-03, 0.3663750368606913e-03, 0.5236148327211592e+00, 0.5059345305621403e+00, 0.2766627272719901e-02, 0.5208666879309463e-02, 0.9525115502267366e+00, 0.1019555750555114e+01, 0.9233342160256496e-01, 0.1271177219084926e+00, 0.1403647802928048e+01, 0.1295158592310706e+01, 0.3870906568602875e-01, 0.2575581550365553e-01, 0.8617584837127339e+00, 0.8333292886837029e+00, 0.1740717486950262e-01, 0.2915663048658400e-01, 0.1395808011693333e+01, 0.1558759230767007e+01, 0.2121884039551361e+00, 0.8569226352498945e-01, 0.6295047433559955e+00, 0.1058885473974269e+01, 0.5351535485381410e+00, 0.1868951540728986e+00, 0.3698014604305115e+00, 0.6910236107191923e+00, 0.5281978587564573e+00, 0.1000437033056103e+01, 0.1308838344228090e+01, 0.3918007450464985e+00, 0.1034799882693896e+00, 0.2379777816619179e+00, 0.9010425466701916e+00, 0.6771983450302391e+00, 0.3421413441684364e-01, 0.4045895778871807e-01, 0.8008017657953598e+00, 0.1374143583674083e+01, 0.6338813300623194e+00, 0.4005339232303680e+00, 0.8682873190108105e+00, 0.1070641461682245e+01, 0.6229686202966810e+00, 0.4896740201485869e+00, 0.8415597376799326e+00, 0.4351392095057171e+00, 0.1033340463692495e+00, 0.1811694274708021e+00, 0.7629036530181579e+00, 0.1686298765095183e+01, 0.1124645093942705e+01, 0.4654019842229331e+00, 0.6978261813392677e+00, 0.5176639768935063e+00, 0.3053203341720497e+00, 0.2907699806261721e+00, 0.4929941693545132e+00, 0.7331214635495586e+00, 0.5509778292160957e+00, 0.1568680213363420e+00, 0.2087258457383731e+00, 0.8093075758121498e-01, 0.4915348757406203e-01, 0.2778348386000479e-01, 0.4574524632962537e-01, -0.8652862282021575e-03, 0.1253505697055357e-02, -0.3732238779264379e-01, 0.2576338363396270e-01, 0.4300521703172840e-01, 0.8149780515932184e-03, 0.3663750368606913e-03},
			i0:       1,
			n0:       20,
			pp:       0,
			tau:      5.0201385987195205e-003,
			sigma:    2.1175425068616867e-005,
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			tauOut:   5.0201385987195205e-003,
			sigmaOut: 2.1175425068616867e-005,
			dminOut:  1.8576185384092288e-002,
			dmin1Out: 1.8576185384092288e-002,
			dmin2Out: 5.2365600435162571e-002,
			dnOut:    1.9114649409197451e-002,
			dnm1Out:  1.8576185384092288e-002,
			dnm2Out:  5.2365600435162571e-002,
		},
		{
			z:        []float64{0.2471569222413904e+01, 0.2471445466333236e+01, 0.4896382518051712e-02, 0.2350431231346416e-02, 0.1186378431264915e+01, 0.1179822839486443e+01, 0.8149780515932184e-03, 0.3616937915375072e-03, 0.5236148327211592e+00, 0.5209996276036221e+00, 0.2766627272719901e-02, 0.5058054349403302e-02, 0.9525115502267366e+00, 0.1034766778881179e+01, 0.9233342160256496e-01, 0.1252490967185870e+00, 0.1403647802928048e+01, 0.1312087633296770e+01, 0.3870906568602875e-01, 0.2542350442532051e-01, 0.8617584837127339e+00, 0.8487220155581966e+00, 0.1740717486950262e-01, 0.2862783537884150e-01, 0.1395808011693333e+01, 0.1574348441670908e+01, 0.2121884039551361e+00, 0.8484373804386666e-01, 0.6295047433559955e+00, 0.1074794415251550e+01, 0.5351535485381410e+00, 0.1841287608083240e+00, 0.3698014604305115e+00, 0.7088504197799252e+00, 0.5281978587564573e+00, 0.9752771411128711e+00, 0.1308838344228090e+01, 0.4320210527858890e+00, 0.1034799882693896e+00, 0.2158225196628609e+00, 0.9010425466701916e+00, 0.7144140228254550e+00, 0.3421413441684364e-01, 0.3835134583138245e-01, 0.8008017657953598e+00, 0.1391311611427577e+01, 0.6338813300623194e+00, 0.3955915526975877e+00, 0.8682873190108105e+00, 0.1090644248011184e+01, 0.6229686202966810e+00, 0.4806932321292802e+00, 0.8415597376799326e+00, 0.4591804133211825e+00, 0.1033340463692495e+00, 0.1716839812178710e+00, 0.7629036530181579e+00, 0.1710844627144272e+01, 0.1124645093942705e+01, 0.4587247601659613e+00, 0.6978261813392677e+00, 0.5394016167466366e+00, 0.3053203341720497e+00, 0.2790520826393697e+00, 0.4929941693545132e+00, 0.7598997773325197e+00, 0.5509778292160957e+00, 0.1513401067044909e+00, 0.2087258457383731e+00, 0.1015190880092246e+00, 0.4915348757406203e-01, 0.2214892234681356e-01, 0.4574524632962537e-01, 0.1982969108114764e-01, 0.1253505697055357e-02, 0.1628595626045726e-02, 0.2576338363396270e-01, 0.1911464940919745e-01, 0.8149780515932184e-03, 0.3616937915375072e-03, 0.2117542506861687e-04},
			i0:       1,
			n0:       20,
			pp:       1,
			tau:      5.4769133315280185e-003,
			sigma:    5.0413140237881371e-003,
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			tauOut:   5.4769133315280185e-003,
			sigmaOut: 5.0413140237881371e-003,
			dminOut:  9.5622848228283271e-003,
			dmin1Out: 9.5622848228283271e-003,
			dmin2Out: 6.9533978479808370e-002,
			dnOut:    1.0856003705186750e-002,
			dnm1Out:  9.5622848228283271e-003,
			dnm2Out:  6.9533978479808370e-002,
		},
		{
			z:        []float64{0.2468318984233055e+01, 0.2471445466333236e+01, 0.1123474100024551e-02, 0.2350431231346416e-02, 0.1173584145846428e+01, 0.1179822839486443e+01, 0.1605699355811189e-03, 0.3616937915375072e-03, 0.5204201986859162e+00, 0.5209996276036221e+00, 0.1005707814522541e-01, 0.5058054349403302e-02, 0.1144481884123012e+01, 0.1034766778881179e+01, 0.1435914304680996e+00, 0.1252490967185870e+00, 0.1188442793922463e+01, 0.1312087633296770e+01, 0.1815610143690141e-01, 0.2542350442532051e-01, 0.8537168361686087e+00, 0.8487220155581966e+00, 0.5279290053521807e-01, 0.2862783537884150e-01, 0.1600922365848029e+01, 0.1574348441670908e+01, 0.5696064828871891e-01, 0.8484373804386666e-01, 0.1196485614439627e+01, 0.1074794415251550e+01, 0.1090859328498209e+00, 0.1841287608083240e+00, 0.1569564714711448e+01, 0.7088504197799252e+00, 0.2684440171930437e+00, 0.9752771411128711e+00, 0.3739226419241781e+00, 0.4320210527858890e+00, 0.4123490187575627e+00, 0.2158225196628609e+00, 0.3349394365677468e+00, 0.7144140228254550e+00, 0.1593084209965356e+00, 0.3835134583138245e-01, 0.1622117829797102e+01, 0.1391311611427577e+01, 0.2659792301064862e+00, 0.3955915526975877e+00, 0.1299881336702450e+01, 0.1090644248011184e+01, 0.1698038973078534e+00, 0.4806932321292802e+00, 0.4555835838996722e+00, 0.4591804133211825e+00, 0.6447216871142054e+00, 0.1716839812178710e+00, 0.1519370786864500e+01, 0.1710844627144272e+01, 0.1628548339973444e+00, 0.4587247601659613e+00, 0.6501219520571339e+00, 0.5394016167466366e+00, 0.3261720586281595e+00, 0.2790520826393697e+00, 0.5795909120773233e+00, 0.7598997773325197e+00, 0.2650819619788820e-01, 0.1513401067044909e+00, 0.9168290082662192e-01, 0.1015190880092246e+00, 0.4790492926791300e-02, 0.2214892234681356e-01, 0.1119088044887405e-01, 0.1982969108114764e-01, 0.2781732372482683e-02, 0.1628595626045726e-02, 0.1085600370518675e-01, 0.1911464940919745e-01, 0.1605699355811189e-03, 0.3616937915375072e-03},
			i0:       1,
			n0:       20,
			pp:       0,
			tau:      5.1216063611655054e-003,
			sigma:    1.0518227355316156e-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			tauOut:   5.1216063611655054e-003,
			sigmaOut: 1.0518227355316156e-002,
			dminOut:  2.0601312480394186e-003,
			dmin1Out: 5.4371870398960158e-003,
			dmin2Out: 8.0023511442426670e-002,
			dnOut:    2.0601312480394186e-003,
			dnm1Out:  5.4371870398960158e-003,
			dnm2Out:  8.0023511442426670e-002,
		},
		{
			z:        []float64{0.2468318984233055e+01, 0.2464320851971913e+01, 0.1123474100024551e-02, 0.5350323562789559e-03, 0.1173584145846428e+01, 0.1168088077064565e+01, 0.1605699355811189e-03, 0.7153898701552432e-04, 0.5204201986859162e+00, 0.5252841314829605e+00, 0.1005707814522541e-01, 0.2191222436498315e-01, 0.1144481884123012e+01, 0.1261039483864963e+01, 0.1435914304680996e+00, 0.1353250258951489e+00, 0.1188442793922463e+01, 0.1066152263103050e+01, 0.1815610143690141e-01, 0.1453842008528346e-01, 0.8537168361686087e+00, 0.8868497102573779e+00, 0.5279290053521807e-01, 0.9530062900995111e-01, 0.1600922365848029e+01, 0.1557460778765631e+01, 0.5696064828871891e-01, 0.4375878814786067e-01, 0.1196485614439627e+01, 0.1256691152780422e+01, 0.1090859328498209e+00, 0.1362446379077657e+00, 0.1569564714711448e+01, 0.1696642487635560e+01, 0.2684440171930437e+00, 0.5916231430550117e-01, 0.3739226419241781e+00, 0.7219877400150740e+00, 0.4123490187575627e+00, 0.1912940350054112e+00, 0.3349394365677468e+00, 0.2978322161977056e+00, 0.1593084209965356e+00, 0.8676597630518320e+00, 0.1622117829797102e+01, 0.1015315690490590e+01, 0.2659792301064862e+00, 0.3405260456467969e+00, 0.1299881336702450e+01, 0.1124037582002341e+01, 0.1698038973078534e+00, 0.6882320425428856e-01, 0.4555835838996722e+00, 0.1026360460398424e+01, 0.6447216871142054e+00, 0.9544125430154021e+00, 0.1519370786864500e+01, 0.7226914714852769e+00, 0.1628548339973444e+00, 0.1465016632377001e+00, 0.6501219520571339e+00, 0.8246707410864278e+00, 0.3261720586281595e+00, 0.2292385937027206e+00, 0.5795909120773233e+00, 0.3717389082113253e+00, 0.2650819619788820e-01, 0.6537783023029759e-02, 0.9168290082662192e-01, 0.8481400436921797e-01, 0.4790492926791300e-02, 0.6320870478125323e-03, 0.1119088044887405e-01, 0.8218919412378699e-02, 0.2781732372482683e-02, 0.3674266095981827e-02, 0.1085600370518675e-01, 0.2060131248039419e-02, 0.1605699355811189e-03, 0.7153898701552432e-04, 0.2117542506861687e-04},
			i0:       1,
			n0:       20,
			pp:       1,
			tau:      1.2817878169164906e-003,
			sigma:    1.5639833716481661e-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			tauOut:   1.2817878169164906e-003,
			sigmaOut: 1.5639833716481661e-002,
			dminOut:  6.0731361153288982e-005,
			dmin1Out: 6.8738708636769136e-003,
			dmin2Out: 8.1489512892123819e-002,
			dnOut:    6.0731361153288982e-005,
			dnm1Out:  6.8738708636769136e-003,
			dnm2Out:  8.1489512892123819e-002,
		},
		{
			z:        []float64{0.2463574096511276e+01, 0.2464320851971913e+01, 0.2536822079344948e-03, 0.5350323562789559e-03, 0.1166624146026729e+01, 0.1168088077064565e+01, 0.3221114082852138e-04, 0.7153898701552432e-04, 0.5458823568901986e+00, 0.5252841314829605e+00, 0.5061929508212644e-01, 0.2191222436498315e-01, 0.1344463426861069e+01, 0.1261039483864963e+01, 0.1073120173669855e+00, 0.1353250258951489e+00, 0.9720968780044319e+00, 0.1066152263103050e+01, 0.1326348631702415e-01, 0.1453842008528346e-01, 0.9676050651333883e+00, 0.8868497102573779e+00, 0.1533962535161303e+00, 0.9530062900995111e-01, 0.1446541525580445e+01, 0.1557460778765631e+01, 0.3801569533217738e-01, 0.4375878814786067e-01, 0.1353638307539094e+01, 0.1256691152780422e+01, 0.1707682473962209e+00, 0.1362446379077657e+00, 0.1583754766727924e+01, 0.1696642487635560e+01, 0.2697037855661164e-01, 0.5916231430550117e-01, 0.8850296086469572e+00, 0.7219877400150740e+00, 0.6437471225190403e-01, 0.1912940350054112e+00, 0.1099835479180717e+01, 0.2978322161977056e+00, 0.8009821360646626e+00, 0.8676597630518320e+00, 0.5535778122558079e+00, 0.1015315690490590e+01, 0.6914368034330997e+00, 0.3405260456467969e+00, 0.5001421950066134e+00, 0.1124037582002341e+01, 0.1412346654806686e+00, 0.6882320425428856e-01, 0.1838256550116241e+01, 0.1026360460398424e+01, 0.3752173792456719e+00, 0.9544125430154021e+00, 0.4926939676603885e+00, 0.7226914714852769e+00, 0.2452143584512202e+00, 0.1465016632377001e+00, 0.8074131885210117e+00, 0.8246707410864278e+00, 0.1055431169003394e+00, 0.2292385937027206e+00, 0.2714517865170992e+00, 0.3717389082113253e+00, 0.2042703660177667e-02, 0.6537783023029759e-02, 0.8212159993993635e-01, 0.8481400436921797e-01, 0.6326073178529442e-04, 0.6320870478125323e-03, 0.1054813695965874e-01, 0.8218919412378699e-02, 0.7176120699696391e-03, 0.3674266095981827e-02, 0.6073136115328898e-04, 0.2060131248039419e-02, 0.3221114082852138e-04, 0.7153898701552432e-04},
			i0:       1,
			n0:       20,
			pp:       0,
			tau:      5.6837241251038845e-005,
			sigma:    1.6921621533398150e-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			tauOut:   5.6837241251038845e-005,
			sigmaOut: 1.6921621533398150e-002,
			dminOut:  3.1568086081919418e-009,
			dmin1Out: 1.0483100129151506e-002,
			dmin2Out: 8.1316774559040517e-002,
			dnOut:    3.1568086081919418e-009,
			dnm1Out:  1.0483100129151506e-002,
			dnm2Out:  8.1316774559040517e-002,
		},
		{
			z:        []float64{0.2463574096511276e+01, 0.2463770941477959e+01, 0.2536822079344948e-03, 0.1201214707955848e-03, 0.1166624146026729e+01, 0.1166479398455512e+01, 0.3221114082852138e-04, 0.1507398544447245e-04, 0.5458823568901986e+00, 0.5964297407456295e+00, 0.5061929508212644e-01, 0.1141052940222717e+00, 0.1344463426861069e+01, 0.1337613312964532e+01, 0.1073120173669855e+00, 0.7798791776646297e-01, 0.9720968780044319e+00, 0.9073156093137420e+00, 0.1326348631702415e-01, 0.1414482062243694e-01, 0.9676050651333883e+00, 0.1106799660785830e+01, 0.1533962535161303e+00, 0.2004825791345134e+00, 0.1446541525580445e+01, 0.1284017804536858e+01, 0.3801569533217738e-01, 0.4007693764646178e-01, 0.1353638307539094e+01, 0.1484272780047602e+01, 0.1707682473962209e+00, 0.1822138285193538e+00, 0.1583754766727924e+01, 0.1428454479523931e+01, 0.2697037855661164e-01, 0.1671007646458111e-01, 0.8850296086469572e+00, 0.9326374071930291e+00, 0.6437471225190403e-01, 0.7591545433480534e-01, 0.1099835479180717e+01, 0.1824845323669324e+01, 0.8009821360646626e+00, 0.2429827519008994e+00, 0.5535778122558079e+00, 0.1001975026546757e+01, 0.6914368034330997e+00, 0.3451350696526060e+00, 0.5001421950066134e+00, 0.2961849535934249e+00, 0.1412346654806686e+00, 0.8765656248686587e+00, 0.1838256550116241e+01, 0.1336851467252003e+01, 0.3752173792456719e+00, 0.1382856239786244e+00, 0.4926939676603885e+00, 0.5995658648917332e+00, 0.2452143584512202e+00, 0.3302211126778973e+00, 0.8074131885210117e+00, 0.5826783555022028e+00, 0.1055431169003394e+00, 0.4916926700063749e-01, 0.2714517865170992e+00, 0.2242683859353883e+00, 0.2042703660177667e-02, 0.7479881396448043e-03, 0.8212159993993635e-01, 0.8138003529082581e-01, 0.6326073178529442e-04, 0.8199589256196194e-05, 0.1054813695965874e-01, 0.1120071219912114e-01, 0.7176120699696391e-03, 0.3890963093641941e-05, 0.6073136115328898e-04, 0.3156808608191942e-08, 0.3221114082852138e-04, 0.1507398544447245e-04, 0.2117542506861687e-04},
			i0:       1,
			n0:       20,
			pp:       1,
			tau:      3.1557121791797713e-009,
			sigma:    1.6978458774649190e-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			tauOut:   3.1557121791797713e-009,
			sigmaOut: 1.6978458774649190e-002,
			dminOut:  6.9684975813366743e-017,
			dmin1Out: 1.1199576261102989e-002,
			dmin2Out: 8.1067607231828140e-002,
			dnOut:    6.9684975813366743e-017,
			dnm1Out:  1.1199576261102989e-002,
			dnm2Out:  8.1067607231828140e-002,
		},
		{
			z:        []float64{0.2463891059793043e+01, 0.2463770941477959e+01, 0.5686908130061341e-04, 0.1201214707955848e-03, 0.1166437600203943e+01, 0.1166479398455512e+01, 0.7707718980490818e-05, 0.1507398544447245e-04, 0.7105273238932086e+00, 0.5964297407456295e+00, 0.2148105431436762e+00, 0.1141052940222717e+00, 0.1200790684431606e+01, 0.1337613312964532e+01, 0.5892755169139442e-01, 0.7798791776646297e-01, 0.8625328750890724e+00, 0.9073156093137420e+00, 0.1815059242254727e-01, 0.1414482062243694e-01, 0.1289131644342084e+01, 0.1106799660785830e+01, 0.1996872873596725e+00, 0.2004825791345134e+00, 0.1124407451667935e+01, 0.1284017804536858e+01, 0.5290351604133232e-01, 0.4007693764646178e-01, 0.1613583089369911e+01, 0.1484272780047602e+01, 0.1613081850537457e+00, 0.1822138285193538e+00, 0.1283856367779054e+01, 0.1428454479523931e+01, 0.1213877407087503e-01, 0.1671007646458111e-01, 0.9964140843012472e+00, 0.9326374071930291e+00, 0.1390325207358455e+00, 0.7591545433480534e-01, 0.1928795551678665e+01, 0.1824845323669324e+01, 0.1262252233392066e+00, 0.2429827519008994e+00, 0.1220884869704444e+01, 0.1001975026546757e+01, 0.8372928285471114e-01, 0.3451350696526060e+00, 0.1089021292451660e+01, 0.2961849535934249e+00, 0.1076046951396362e+01, 0.8765656248686587e+00, 0.3990901366785531e+00, 0.1336851467252003e+01, 0.2077509116934600e+00, 0.1382856239786244e+00, 0.7220360627204584e+00, 0.5995658648917332e+00, 0.2664862668525171e+00, 0.3302211126778973e+00, 0.3653613524946110e+00, 0.5826783555022028e+00, 0.3018138637972599e-01, 0.4916926700063749e-01, 0.1948349845395949e+00, 0.2242683859353883e+00, 0.3124249032854923e-03, 0.7479881396448043e-03, 0.8107580682108434e-01, 0.8138003529082581e-01, 0.1132782305976083e-05, 0.8199589256196194e-05, 0.1120346722419663e-01, 0.1120071219912114e-01, 0.1096359327194516e-11, 0.3890963093641941e-05, 0.6968497581336674e-16, 0.3156808608191942e-08, 0.7707718980490818e-05, 0.1507398544447245e-04},
			i0:       1,
			n0:       20,
			pp:       0,
			tau:      6.9684975806547287e-017,
			sigma:    1.6978461930361368e-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			tauOut:   6.9684975806547287e-017,
			sigmaOut: 1.6978461930361368e-002,
			dminOut:  6.1629758220391547e-032,
			dmin1Out: 1.1203310405167735e-002,
			dmin2Out: 8.0927116373146771e-002,
			dnOut:    6.1629758220391547e-032,
			dnm1Out:  1.1203310405167735e-002,
			dnm2Out:  8.0927116373146771e-002,
		},
		{
			z:        []float64{0.2463891059793043e+01, 0.2463947928874343e+01, 0.5686908130061341e-04, 0.2692193042748079e-04, 0.1166437600203943e+01, 0.1166418385992496e+01, 0.7707718980490818e-05, 0.4695180568393632e-05, 0.7105273238932086e+00, 0.9253331718563164e+00, 0.2148105431436762e+00, 0.2787563517334627e+00, 0.1200790684431606e+01, 0.9809618843895378e+00, 0.5892755169139442e-01, 0.5181337969514327e-01, 0.8625328750890724e+00, 0.8288700878164763e+00, 0.1815059242254727e-01, 0.2822939734392020e-01, 0.1289131644342084e+01, 0.1460589534357837e+01, 0.1996872873596725e+00, 0.1537255119449346e+00, 0.1124407451667935e+01, 0.1023585455764333e+01, 0.5290351604133232e-01, 0.8339725654733963e-01, 0.1613583089369911e+01, 0.1691494017876317e+01, 0.1613081850537457e+00, 0.1224340957564512e+00, 0.1283856367779054e+01, 0.1173561046093478e+01, 0.1213877407087503e-01, 0.1030644761994533e-01, 0.9964140843012472e+00, 0.1125140157417147e+01, 0.1390325207358455e+00, 0.2383394688796517e+00, 0.1928795551678665e+01, 0.1816681306138221e+01, 0.1262252233392066e+00, 0.8482856339700598e-01, 0.1220884869704444e+01, 0.1219785589162149e+01, 0.8372928285471114e-01, 0.7475327847832687e-01, 0.1089021292451660e+01, 0.2090314965369696e+01, 0.1076046951396362e+01, 0.2054425921547012e+00, 0.3990901366785531e+00, 0.4013984562173118e+00, 0.2077509116934600e+00, 0.3737026089221466e+00, 0.7220360627204584e+00, 0.6148197206508288e+00, 0.2664862668525171e+00, 0.1583615157552351e+00, 0.3653613524946110e+00, 0.2371812231191019e+00, 0.3018138637972599e-01, 0.2479281399828426e-01, 0.1948349845395949e+00, 0.1703545954445960e+00, 0.3124249032854923e-03, 0.1486904479375115e-03, 0.8107580682108434e-01, 0.8092824915545274e-01, 0.1132782305976083e-05, 0.1568190288260776e-06, 0.1120346722419663e-01, 0.1120331040626409e-01, 0.1096359327194516e-11, 0.6819392699821255e-26, 0.6968497581336674e-16, 0.6162975822039155e-31, 0.7707718980490818e-05, 0.4695180568393632e-05, 0.2117542506861687e-04},
			i0:       1,
			n0:       20,
			pp:       1,
			tau:      6.1629758220391547e-032,
			sigma:    1.6978461930361441e-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.6978461930361441e-002,
			dminOut:  6.1629758220391547e-032,
			dmin1Out: 1.1203288675083998e-002,
			dmin2Out: 8.0846453894262649e-002,
			dnOut:    6.1629758220391547e-032,
			dnm1Out:  1.1203288675083998e-002,
			dnm2Out:  8.0846453894262649e-002,
		},
		{
			z:        []float64{0.2463974850804771e+01, 0.2463947928874343e+01, 0.1274454348702788e-04, 0.2692193042748079e-04, 0.1166410336629578e+01, 0.1166418385992496e+01, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.1204085798823124e+01, 0.9253331718563164e+00, 0.2271012218143261e+00, 0.2787563517334627e+00, 0.8056740422703550e+00, 0.9809618843895378e+00, 0.5330513126246473e-01, 0.5181337969514327e-01, 0.8037943538979316e+00, 0.8288700878164763e+00, 0.5129615818002433e-01, 0.2822939734392020e-01, 0.1563018888122747e+01, 0.1460589534357837e+01, 0.1006713350698832e+00, 0.1537255119449346e+00, 0.1006311377241790e+01, 0.1023585455764333e+01, 0.1401812239704283e+00, 0.8339725654733963e-01, 0.1673746889662340e+01, 0.1691494017876317e+01, 0.8584564749956700e-01, 0.1224340957564512e+00, 0.1098021846213856e+01, 0.1173561046093478e+01, 0.1056099032774466e-01, 0.1030644761994533e-01, 0.1352918635969054e+01, 0.1125140157417147e+01, 0.3200390963041470e+00, 0.2383394688796517e+00, 0.1581470773231080e+01, 0.1816681306138221e+01, 0.6542811978092533e-01, 0.8482856339700598e-01, 0.1229110747859551e+01, 0.1219785589162149e+01, 0.1271308521106110e+00, 0.7475327847832687e-01, 0.2168626705413786e+01, 0.2090314965369696e+01, 0.3802606466401751e-01, 0.2054425921547012e+00, 0.7370750004754409e+00, 0.4013984562173118e+00, 0.3117182559112661e+00, 0.3737026089221466e+00, 0.4614629804947978e+00, 0.6148197206508288e+00, 0.8139413038408401e-01, 0.1583615157552351e+00, 0.1805799067333021e+00, 0.2371812231191019e+00, 0.2338892446571373e-01, 0.2479281399828426e-01, 0.1471143614268198e+00, 0.1703545954445960e+00, 0.8179526119010886e-04, 0.1486904479375115e-03, 0.8084661071329148e-01, 0.8092824915545274e-01, 0.2173118009582292e-07, 0.1568190288260776e-06, 0.1120328867508400e-01, 0.1120331040626409e-01, 0.3751376363572422e-55, 0.6819392699821255e-26},
			i0:       1,
			n0:       19,
			pp:       0,
			tau:      1.1203284285913290e-002,
			sigma:    1.6978461930361441e-002,
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			tauOut:   1.1203284285913290e-002,
			sigmaOut: 1.6978461930361441e-002,
			dminOut:  8.9043085058082561e-010,
			dmin1Out: 6.9585227169791292e-002,
			dmin2Out: 0.11373841371027797,
			dnOut:    8.9043085058082561e-010,
			dnm1Out:  6.9585227169791292e-002,
			dnm2Out:  0.11373841371027797,
		},
		{
			z:        []float64{0.2463974850804771e+01, 0.2452784311062345e+01, 0.1274454348702788e-04, 0.6060609239813679e-05, 0.1166410336629578e+01, 0.1155204716501079e+01, 0.3724766654883956e-05, 0.3882375624867434e-05, 0.1204085798823124e+01, 0.1419979853975912e+01, 0.2271012218143261e+00, 0.1288536304732592e+00, 0.8056740422703550e+00, 0.7189222587736472e+00, 0.5330513126246473e-01, 0.5959804835594534e-01, 0.8037943538979316e+00, 0.7842891794360973e+00, 0.5129615818002433e-01, 0.1022287011292917e+00, 0.1563018888122747e+01, 0.1550258237777425e+01, 0.1006713350698832e+00, 0.6534828028921505e-01, 0.1006311377241790e+01, 0.1069941036637090e+01, 0.1401812239704283e+00, 0.2192904838448094e+00, 0.1673746889662340e+01, 0.1529098769031184e+01, 0.8584564749956700e-01, 0.6164441320989392e-01, 0.1098021846213856e+01, 0.1035735139045794e+01, 0.1056099032774466e-01, 0.1379518767882891e-01, 0.1352918635969054e+01, 0.1647959260308459e+01, 0.3200390963041470e+00, 0.3071268139247320e+00, 0.1581470773231080e+01, 0.1328568794801360e+01, 0.6542811978092533e-01, 0.6053010243026304e-01, 0.1229110747859551e+01, 0.1284508213253985e+01, 0.1271308521106110e+00, 0.2146341752620367e+00, 0.2168626705413786e+01, 0.1980815310529854e+01, 0.3802606466401751e-01, 0.1414976019284330e-01, 0.7370750004754409e+00, 0.1023440211907950e+01, 0.3117182559112661e+00, 0.1405518698344743e+00, 0.4614629804947978e+00, 0.3911019567584944e+00, 0.8139413038408401e-01, 0.3758136265851575e-01, 0.1805799067333021e+00, 0.1551841842545868e+00, 0.2338892446571373e-01, 0.2217266343062855e-01, 0.1471143614268198e+00, 0.1138202089714681e+00, 0.8179526119010886e-04, 0.5809925758690545e-04, 0.8084661071329148e-01, 0.6958524890097138e-01, 0.2173118009582292e-07, 0.3498739858072780e-08, 0.1120328867508400e-01, 0.8904308505808256e-09, 0.3751376363572422e-55, 0.3882375624867434e-05, 0.1697846193036144e-01},
			i0:       1,
			n0:       19,
			pp:       1,
			tau:      8.9043080564395014e-010,
			sigma:    2.8181746216274728e-002,
			i0Out:    1,
			n0Out:    19,
			ppOut:    1,
			tauOut:   8.9043080564395014e-010,
			sigmaOut: 2.8181746216274728e-002,
			dminOut:  1.3962105636374437e-019,
			dmin1Out: 6.9544122471813200e-002,
			dmin2Out: 9.8247027562672340e-002,
			dnOut:    1.3962105636374437e-019,
			dnm1Out:  6.9544122471813200e-002,
			dnm2Out:  9.8247027562672340e-002,
		},
		{
			z:        []float64{0.2452790370781154e+01, 0.2452784311062345e+01, 0.2854399814229969e-05, 0.6060609239813679e-05, 0.1155205743586459e+01, 0.1155204716501079e+01, 0.4772219324121025e-05, 0.3882375624867434e-05, 0.1548828711339416e+01, 0.1419979853975912e+01, 0.5981019230390531e-01, 0.1288536304732592e+00, 0.7187101139352565e+00, 0.7189222587736472e+00, 0.6503610222645050e-01, 0.5959804835594534e-01, 0.8214817774485077e+00, 0.7842891794360973e+00, 0.1929207566298143e+00, 0.1022287011292917e+00, 0.1422685760546395e+01, 0.1550258237777425e+01, 0.4914564318703864e-01, 0.6534828028921505e-01, 0.1240085876404429e+01, 0.1069941036637090e+01, 0.2703980549150243e+00, 0.2192904838448094e+00, 0.1320345126435623e+01, 0.1529098769031184e+01, 0.4835651195207321e-01, 0.6164441320989392e-01, 0.1001173813882118e+01, 0.1035735139045794e+01, 0.2270725319399537e-01, 0.1379518767882891e-01, 0.1932378820148765e+01, 0.1647959260308459e+01, 0.2111589595024383e+00, 0.3071268139247320e+00, 0.1177939936838754e+01, 0.1328568794801360e+01, 0.6600626338337756e-01, 0.6053010243026304e-01, 0.1433136124242214e+01, 0.1284508213253985e+01, 0.2966575563411978e+00, 0.2146341752620367e+00, 0.1698307513491068e+01, 0.1980815310529854e+01, 0.8526979628348903e-02, 0.1414976019284330e-01, 0.1155465101223645e+01, 0.1023440211907950e+01, 0.4757401262929915e-01, 0.1405518698344743e+00, 0.3811093058972801e+00, 0.3911019567584944e+00, 0.1530278326215801e-01, 0.3758136265851575e-01, 0.1620540635326265e+00, 0.1551841842545868e+00, 0.1557318051836492e-01, 0.2217266343062855e-01, 0.9830512682025924e-01, 0.1138202089714681e+00, 0.4112553872737890e-04, 0.5809925758690545e-04, 0.6954412597055305e-01, 0.6958524890097138e-01, 0.4479725446695418e-16, 0.3498739858072780e-08, 0.1396210563637444e-18, 0.8904308505808256e-09, 0.2854399814229969e-05, 0.3882375624867434e-05},
			i0:       1,
			n0:       19,
			pp:       0,
			tau:      1.3962105636374427e-019,
			sigma:    2.8181747106705537e-002,
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 2.8181747106705537e-002,
			dminOut:  1.3962105636374430e-019,
			dmin1Out: 6.9512134440652351e-002,
			dmin2Out: 8.9358776708560295e-002,
			dnOut:    1.3962105636374430e-019,
			dnm1Out:  6.9512134440652351e-002,
			dnm2Out:  8.9358776708560295e-002,
		},
		{
			z:        []float64{0.2452790370781154e+01, 0.2452793225180968e+01, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1155205743586459e+01, 0.1155209171453138e+01, 0.4772219324121025e-05, 0.6398278760815103e-05, 0.1548828711339416e+01, 0.1608632505364561e+01, 0.5981019230390531e-01, 0.2672219415054498e-01, 0.7187101139352565e+00, 0.7570240220111620e+00, 0.6503610222645050e-01, 0.7057368234283014e-01, 0.8214817774485077e+00, 0.9438288517354918e+00, 0.1929207566298143e+00, 0.2908001941945216e+00, 0.1422685760546395e+01, 0.1181031209538913e+01, 0.4914564318703864e-01, 0.5160305461093759e-01, 0.1240085876404429e+01, 0.1458880876708516e+01, 0.2703980549150243e+00, 0.2447209773632920e+00, 0.1320345126435623e+01, 0.1123980661024405e+01, 0.4835651195207321e-01, 0.4307304847484577e-01, 0.1001173813882118e+01, 0.9808080186012680e+00, 0.2270725319399537e-01, 0.4473761868138884e-01, 0.1932378820148765e+01, 0.2098800160969815e+01, 0.2111589595024383e+00, 0.1185117935689049e+00, 0.1177939936838754e+01, 0.1125434406653226e+01, 0.6600626338337756e-01, 0.8405284210411720e-01, 0.1433136124242214e+01, 0.1645740838479294e+01, 0.2966575563411978e+00, 0.3061331074057168e+00, 0.1698307513491068e+01, 0.1400701385713701e+01, 0.8526979628348903e-02, 0.7034066989504621e-02, 0.1155465101223645e+01, 0.1196005046863439e+01, 0.4757401262929915e-01, 0.1515955052150448e-01, 0.3811093058972801e+00, 0.3812525386379337e+00, 0.1530278326215801e-01, 0.6504555274179689e-02, 0.1620540635326265e+00, 0.1711226887768117e+00, 0.1557318051836492e-01, 0.8946350111698947e-02, 0.9830512682025924e-01, 0.8939990224728768e-01, 0.4112553872737890e-04, 0.3199152990069927e-04, 0.6954412597055305e-01, 0.6951213444065239e-01, 0.4479725446695418e-16, 0.8997910999570377e-34, 0.2818174710670554e-01},
			i0:       1,
			n0:       18,
			pp:       1,
			tau:      6.8088572105595116e-002,
			sigma:    2.8181747106705537e-002,
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			tauOut:   6.8088572105595116e-002,
			sigmaOut: 2.8181747106705537e-002,
			dminOut:  1.2643548659038301e-003,
			dmin1Out: 1.3935943135754067e-002,
			dmin2Out: 9.9495800084882416e-002,
			dnOut:    1.2643548659038301e-003,
			dnm1Out:  1.3935943135754067e-002,
			dnm2Out:  9.9495800084882416e-002,
		},
		{
			z:        []float64{0.2384705997428018e+01, 0.2452793225180968e+01, 0.6512368848977837e-06, 0.1344352644992036e-05, 0.1087126346389419e+01, 0.1155209171453138e+01, 0.9467601652019932e-05, 0.6398278760815103e-05, 0.1567256659807859e+01, 0.1608632505364561e+01, 0.1290748567965171e-01, 0.2672219415054498e-01, 0.7466016465687454e+00, 0.7570240220111620e+00, 0.8921689079377818e-01, 0.7057368234283014e-01, 0.1077323583030640e+01, 0.9438288517354918e+00, 0.3187938243378626e+00, 0.2908001941945216e+00, 0.8457518677063927e+00, 0.1181031209538913e+01, 0.8901276181133644e-01, 0.5160305461093759e-01, 0.1546500520154876e+01, 0.1458880876708516e+01, 0.1778606876095877e+00, 0.2447209773632920e+00, 0.9211044497840675e+00, 0.1123980661024405e+01, 0.4586493023634136e-01, 0.4307304847484577e-01, 0.9115921349407204e+00, 0.9808080186012680e+00, 0.1030014605117353e+00, 0.4473761868138884e-01, 0.2046221921921390e+01, 0.2098800160969815e+01, 0.6518220171905387e-01, 0.1185117935689049e+00, 0.1076216474932694e+01, 0.1125434406653226e+01, 0.1285328723941422e+00, 0.8405284210411720e-01, 0.1755252501385274e+01, 0.1645740838479294e+01, 0.2442959445536252e+00, 0.3061331074057168e+00, 0.1095350936043985e+01, 0.1400701385713701e+01, 0.7680442260639305e-02, 0.7034066989504621e-02, 0.1135395583018709e+01, 0.1196005046863439e+01, 0.5090399511302622e-02, 0.1515955052150448e-01, 0.3145781222952156e+00, 0.3812525386379337e+00, 0.3538316586334169e-02, 0.6504555274179689e-02, 0.1084421501965814e+00, 0.1711226887768117e+00, 0.7375387005938499e-02, 0.8946350111698947e-02, 0.1396793466565477e-01, 0.8939990224728768e-01, 0.1592074691534444e-03, 0.3199152990069927e-04, 0.1264354865903830e-02, 0.6951213444065239e-01, 0.6512368848977837e-06, 0.1344352644992036e-05},
			i0:       1,
			n0:       18,
			pp:       0,
			tau:      1.2463174543591322e-003,
			sigma:    9.6270319212300656e-002,
			i0Out:    1,
			n0Out:    18,
			ppOut:    0,
			tauOut:   1.2463174543591322e-003,
			sigmaOut: 9.6270319212300656e-002,
			dminOut:  1.2236274461701734e-006,
			dmin1Out: 1.1812799375517376e-002,
			dmin2Out: 0.10597947881780349,
			dnOut:    1.2236274461701734e-006,
			dnm1Out:  1.1812799375517376e-002,
			dnm2Out:  0.10597947881780349,
		},
		{
			z:        []float64{0.2384705997428018e+01, 0.2383460331210544e+01, 0.6512368848977837e-06, 0.2970373645586866e-06, 0.1087126346389419e+01, 0.1085889199499347e+01, 0.9467601652019932e-05, 0.1366452649899944e-04, 0.1567256659807859e+01, 0.1578904163506652e+01, 0.1290748567965171e-01, 0.6103442048115082e-02, 0.7466016465687454e+00, 0.8284687778600495e+00, 0.8921689079377818e-01, 0.1160157908485996e+00, 0.1077323583030640e+01, 0.1278855299065544e+01, 0.3187938243378626e+00, 0.2108295383723412e+00, 0.8457518677063927e+00, 0.7226887736910288e+00, 0.8901276181133644e-01, 0.1904807262171572e+00, 0.1546500520154876e+01, 0.1532634164092948e+01, 0.1778606876095877e+00, 0.1068932656188067e+00, 0.9211044497840675e+00, 0.8588297969472432e+00, 0.4586493023634136e-01, 0.4868264913684867e-01, 0.9115921349407204e+00, 0.9646646288612479e+00, 0.1030014605117353e+00, 0.2184840619043245e+00, 0.2046221921921390e+01, 0.1891673744281760e+01, 0.6518220171905387e-01, 0.3708364593761750e-01, 0.1076216474932694e+01, 0.1166419383934860e+01, 0.1285328723941422e+00, 0.1934189785315259e+00, 0.1755252501385274e+01, 0.1804883149953014e+01, 0.2442959445536252e+00, 0.1482587897978486e+00, 0.1095350936043985e+01, 0.9535262710524167e+00, 0.7680442260639305e-02, 0.9145359161143373e-02, 0.1135395583018709e+01, 0.1130094305914509e+01, 0.5090399511302622e-02, 0.1416986451145964e-02, 0.3145781222952156e+00, 0.3154531349760448e+00, 0.3538316586334169e-02, 0.1216353924418735e-02, 0.1084421501965814e+00, 0.1133548658237420e+00, 0.7375387005938499e-02, 0.9088178357782586e-03, 0.1396793466565477e-01, 0.1197200684467082e-01, 0.1592074691534444e-03, 0.1681378409852773e-04, 0.1264354865903830e-02, 0.1223627446170173e-05, 0.6512368848977837e-06, 0.2970373645586866e-06, 0.2818174710670554e-01},
			i0:       1,
			n0:       18,
			pp:       1,
			tau:      1.2218877946904154e-006,
			sigma:    9.7516636666659787e-002,
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			tauOut:   1.2218877946904154e-006,
			sigmaOut: 9.7516636666659787e-002,
			dminOut:  9.5986727915820745e-012,
			dmin1Out: 1.1875197612370336e-002,
			dmin2Out: 0.11291768719739126,
			dnOut:    9.5986727915820745e-012,
			dnm1Out:  1.1875197612370336e-002,
			dnm2Out:  0.11291768719739126,
		},
		{
			z:        []float64{0.2383459406360114e+01, 0.2383460331210544e+01, 0.1353283656358167e-06, 0.2970373645586866e-06, 0.1085901506809686e+01, 0.1085889199499347e+01, 0.1986826396898849e-04, 0.1366452649899944e-04, 0.1584986515403004e+01, 0.1578904163506652e+01, 0.3190255011763210e-02, 0.6103442048115082e-02, 0.9412930918090910e+00, 0.8284687778600495e+00, 0.1576208411525267e+00, 0.1160157908485996e+00, 0.1332062774397564e+01, 0.1278855299065544e+01, 0.1143821023097510e+00, 0.2108295383723412e+00, 0.7987861757106404e+00, 0.7226887736910288e+00, 0.3654761154847582e+00, 0.1904807262171572e+00, 0.1274050092339202e+01, 0.1532634164092948e+01, 0.7205613198290631e-01, 0.1068932656188067e+00, 0.8354550922133908e+00, 0.8588297969472432e+00, 0.5621179414582513e-01, 0.4868264913684867e-01, 0.1126935674731953e+01, 0.9646646288612479e+00, 0.3667472533840468e+00, 0.2184840619043245e+00, 0.1562008914947536e+01, 0.1891673744281760e+01, 0.2769195683500124e-01, 0.3708364593761750e-01, 0.1332145183743590e+01, 0.1166419383934860e+01, 0.2620575140703801e+00, 0.1934189785315259e+00, 0.1691083203792688e+01, 0.1804883149953014e+01, 0.8359650824372877e-01, 0.1482587897978486e+00, 0.8790739000820365e+00, 0.9535262710524167e+00, 0.1175682535061811e-01, 0.9145359161143373e-02, 0.1119753245127243e+01, 0.1130094305914509e+01, 0.3991886785572840e-03, 0.1416986451145964e-02, 0.3162690783341116e+00, 0.3154531349760448e+00, 0.4359567385560517e-03, 0.1216353924418735e-02, 0.1138265050331695e+00, 0.1133548658237420e+00, 0.9558734450579545e-04, 0.9088178357782586e-03, 0.1189201139646886e-01, 0.1197200684467082e-01, 0.1730052806966466e-08, 0.1681378409852773e-04, 0.9598672791582074e-11, 0.1223627446170173e-05, 0.1353283656358167e-06, 0.2970373645586866e-06},
			i0:       1,
			n0:       18,
			pp:       0,
			tau:      9.5986713933289272e-012,
			sigma:    9.7517858554454467e-002,
			i0Out:    1,
			n0Out:    18,
			ppOut:    0,
			tauOut:   9.5986713933289272e-012,
			sigmaOut: 9.7517858554454467e-002,
			dminOut:  6.6174125886815435e-022,
			dmin1Out: 1.1882019541390456e-002,
			dmin2Out: 0.11366976186212303,
			dnOut:    6.6174125886815435e-022,
			dnm1Out:  1.1882019541390456e-002,
			dnm2Out:  0.11366976186212303,
		},
		{
			z:        []float64{0.2383459406360114e+01, 0.2383459541678881e+01, 0.1353283656358167e-06, 0.6165545233233256e-07, 0.1085901506809686e+01, 0.1085921313408604e+01, 0.1986826396898849e-04, 0.2899927470478232e-04, 0.1584986515403004e+01, 0.1588147771130463e+01, 0.3190255011763210e-02, 0.1890859942802736e-02, 0.9412930918090910e+00, 0.1097023073009216e+01, 0.1576208411525267e+00, 0.1913914667196325e+00, 0.1332062774397564e+01, 0.1255053409978084e+01, 0.1143821023097510e+00, 0.7279916643176543e-01, 0.7987861757106404e+00, 0.1091463124754034e+01, 0.3654761154847582e+00, 0.4266153094141972e+00, 0.1274050092339202e+01, 0.9194909148983121e+00, 0.7205613198290631e-01, 0.6547064404326042e-01, 0.8354550922133908e+00, 0.8261962423063568e+00, 0.5621179414582513e-01, 0.7667315937770838e-01, 0.1126935674731953e+01, 0.1417009768728692e+01, 0.3667472533840468e+00, 0.4042756034295816e+00, 0.1562008914947536e+01, 0.1185425268343357e+01, 0.2769195683500124e-01, 0.3111938635974581e-01, 0.1332145183743590e+01, 0.1563083311444625e+01, 0.2620575140703801e+00, 0.2835172362390010e+00, 0.1691083203792688e+01, 0.1491162475787817e+01, 0.8359650824372877e-01, 0.4928202642453804e-01, 0.8790739000820365e+00, 0.8415486989985177e+00, 0.1175682535061811e-01, 0.1564347179719430e-01, 0.1119753245127243e+01, 0.1104508961999007e+01, 0.3991886785572840e-03, 0.1143051254380294e-03, 0.3162690783341116e+00, 0.3165907299376309e+00, 0.4359567385560517e-03, 0.1567431614478116e-03, 0.1138265050331695e+00, 0.1137653492066288e+00, 0.9558734450579545e-04, 0.9991845479738400e-05, 0.1189201139646886e-01, 0.1188202127144326e-01, 0.1730052806966466e-08, 0.1397591405272086e-17, 0.9598672791582074e-11, 0.6617412588681544e-21, 0.1353283656358167e-06, 0.6165545233233256e-07, 0.2818174710670554e-01},
			i0:       1,
			n0:       18,
			pp:       1,
			tau:      6.6174125886815426e-022,
			sigma:    9.7517858564053150e-002,
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 9.7517858564053150e-002,
			dminOut:  6.6174125886815426e-022,
			dmin1Out: 1.1880977265773463e-002,
			dmin2Out: 0.11370904624558997,
			dnOut:    6.6174125886815426e-022,
			dnm1Out:  1.1880977265773463e-002,
			dnm2Out:  0.11370904624558997,
		},
		{
			z:        []float64{0.2383459603334333e+01, 0.2383459541678881e+01, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.1085950284592642e+01, 0.1085921313408604e+01, 0.4240998334843304e-04, 0.2899927470478232e-04, 0.1589996221089918e+01, 0.1588147771130463e+01, 0.1304604978030436e-02, 0.1890859942802736e-02, 0.1287109934750819e+01, 0.1097023073009216e+01, 0.1866247058326724e+00, 0.1913914667196325e+00, 0.1141227870577177e+01, 0.1255053409978084e+01, 0.6962466280543776e-01, 0.7279916643176543e-01, 0.1448453771362794e+01, 0.1091463124754034e+01, 0.2708190685256155e+00, 0.4266153094141972e+00, 0.7141424904159570e+00, 0.9194909148983121e+00, 0.7574342769943966e-01, 0.6547064404326042e-01, 0.8271259739846255e+00, 0.8261962423063568e+00, 0.1313543755784947e+00, 0.7667315937770838e-01, 0.1689930996579779e+01, 0.1417009768728692e+01, 0.2835846650840220e+00, 0.4042756034295816e+00, 0.9329599896190804e+00, 0.1185425268343357e+01, 0.5213749144931325e-01, 0.3111938635974581e-01, 0.1794463056234313e+01, 0.1563083311444625e+01, 0.2355970842920850e+00, 0.2835172362390010e+00, 0.1304847417920270e+01, 0.1491162475787817e+01, 0.3178396542921673e-01, 0.4928202642453804e-01, 0.8254082053664953e+00, 0.8415486989985177e+00, 0.2093310277804656e-01, 0.1564347179719430e-01, 0.1083690164346398e+01, 0.1104508961999007e+01, 0.3339325601415243e-04, 0.1143051254380294e-03, 0.3167140798430647e+00, 0.3165907299376309e+00, 0.5630296103885627e-04, 0.1567431614478116e-03, 0.1137190380910697e+00, 0.1137653492066288e+00, 0.1044005669799533e-05, 0.9991845479738400e-05, 0.1188097726577346e-01, 0.1188202127144326e-01, 0.7784240935906335e-37, 0.1397591405272086e-17},
			i0:       1,
			n0:       17,
			pp:       0,
			tau:      1.1880832155707781e-002,
			sigma:    9.7517858564053150e-002,
			i0Out:    1,
			n0Out:    17,
			ppOut:    0,
			tauOut:   1.1880832155707781e-002,
			sigmaOut: 9.7517858564053150e-002,
			dminOut:  2.3287035572430725e-008,
			dmin1Out: 0.10181720511285566,
			dmin2Out: 0.30482311636216664,
			dnOut:    2.3287035572430725e-008,
			dnm1Out:  0.10181720511285566,
			dnm2Out:  0.30482311636216664,
		},
		{
			z:        []float64{0.2383459603334333e+01, 0.2371578799269292e+01, 0.2809066689524106e-07, 0.1286276792433928e-07, 0.1085950284592642e+01, 0.1074111849557515e+01, 0.4240998334843304e-04, 0.6277904232066118e-04, 0.1589996221089918e+01, 0.1579357214869920e+01, 0.1304604978030436e-02, 0.1063198377376995e-02, 0.1287109934750819e+01, 0.1460790610050406e+01, 0.1866247058326724e+00, 0.1457986614708343e+00, 0.1141227870577177e+01, 0.1053173039756072e+01, 0.6962466280543776e-01, 0.9575644420574685e-01, 0.1448453771362794e+01, 0.1611635563526955e+01, 0.2708190685256155e+00, 0.1200044280642223e+00, 0.7141424904159570e+00, 0.6580006578954666e+00, 0.7574342769943966e-01, 0.9521169265880262e-01, 0.8271259739846255e+00, 0.8513878247486099e+00, 0.1313543755784947e+00, 0.2607270439790754e+00, 0.1689930996579779e+01, 0.1700907785529018e+01, 0.2835846650840220e+00, 0.1555482010511415e+00, 0.9329599896190804e+00, 0.8176684478615445e+00, 0.5213749144931325e-01, 0.1144214412274440e+00, 0.1794463056234313e+01, 0.1903757867143247e+01, 0.2355970842920850e+00, 0.1614796988702029e+00, 0.1304847417920270e+01, 0.1163270852323576e+01, 0.3178396542921673e-01, 0.2255256874351997e-01, 0.8254082053664953e+00, 0.8119079072453143e+00, 0.2093310277804656e-01, 0.2794035799797570e-01, 0.1083690164346398e+01, 0.1043902367448729e+01, 0.3339325601415243e-04, 0.1013132519024162e-04, 0.3167140798430647e+00, 0.3048794193232055e+00, 0.5630296103885627e-04, 0.2100082250625822e-04, 0.1137190380910697e+00, 0.1018182491185255e+00, 0.1044005669799533e-05, 0.1218230301111122e-06, 0.1188097726577346e-01, 0.2328703557243073e-07, 0.7784240935906335e-37, 0.1286276792433928e-07, 0.9751785856405315e-01},
			i0:       1,
			n0:       17,
			pp:       1,
			tau:      2.3287007705477136e-008,
			sigma:    0.10939869071976092,
			i0Out:    1,
			n0Out:    17,
			ppOut:    1,
			tauOut:   2.3287007705477136e-008,
			sigmaOut: 0.10939869071976092,
			dminOut:  2.6961654281145418e-018,
			dmin1Out: 0.10181121275944585,
			dmin2Out: 0.30487633314418217,
			dnOut:    2.6961654281145418e-018,
			dnm1Out:  0.10181121275944585,
			dnm2Out:  0.30487633314418217,
		},
		{
			z:        []float64{0.2371578788845052e+01, 0.2371578799269292e+01, 0.5825676764620370e-08, 0.1286276792433928e-07, 0.1074174599487151e+01, 0.1074111849557515e+01, 0.9230392664199863e-04, 0.6277904232066118e-04, 0.1580328086033647e+01, 0.1579357214869920e+01, 0.9827770701659698e-03, 0.1063198377376995e-02, 0.1605606471164067e+01, 0.1460790610050406e+01, 0.9563440497488795e-01, 0.1457986614708343e+00, 0.1053295055699924e+01, 0.1053173039756072e+01, 0.1465159169633776e+00, 0.9575644420574685e-01, 0.1585124051340792e+01, 0.1611635563526955e+01, 0.4981502397231050e-01, 0.1200044280642223e+00, 0.7033973032949510e+00, 0.6580006578954666e+00, 0.1152436546510613e+00, 0.9521169265880262e-01, 0.9968711907896164e+00, 0.8513878247486099e+00, 0.4448645553200346e+00, 0.2607270439790754e+00, 0.1411591407973117e+01, 0.1700907785529018e+01, 0.9010174998427344e-01, 0.1555482010511415e+00, 0.8419881158177074e+00, 0.8176684478615445e+00, 0.2587099684834221e+00, 0.1144214412274440e+00, 0.1806527574243020e+01, 0.1903757867143247e+01, 0.1039810460775319e+00, 0.1614796988702029e+00, 0.1081842351702556e+01, 0.1163270852323576e+01, 0.1692539477932339e-01, 0.2255256874351997e-01, 0.8229228471769590e+00, 0.8119079072453143e+00, 0.3544318396494814e-01, 0.2794035799797570e-01, 0.1008469291521964e+01, 0.1043902367448729e+01, 0.3062892015595061e-05, 0.1013132519024162e-04, 0.3048973339666884e+00, 0.3048794193232055e+00, 0.7013072071892081e-05, 0.2100082250625822e-04, 0.1018113345824760e+00, 0.1018182491185255e+00, 0.2786425742647189e-13, 0.1218230301111122e-06, 0.2696165428114542e-17, 0.2328703557243073e-07, 0.5825676764620370e-08, 0.1286276792433928e-07},
			i0:       1,
			n0:       17,
			pp:       0,
			tau:      2.6961654281138038e-018,
			sigma:    0.10939871400676864,
			i0Out:    1,
			n0Out:    17,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.10939871400676864,
			dminOut:  2.6961654281138038e-018,
			dmin1Out: 0.10180899282350273,
			dmin2Out: 0.30489636738914000,
			dnOut:    2.6961654281138038e-018,
			dnm1Out:  0.10180899282350273,
			dnm2Out:  0.30489636738914000,
		},
		{
			z:        []float64{0.2371578788845052e+01, 0.2371578794670729e+01, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.1074174599487151e+01, 0.1074266900775131e+01, 0.9230392664199863e-04, 0.1357860766428602e-03, 0.1580328086033647e+01, 0.1581175077027170e+01, 0.9827770701659698e-03, 0.9979623676695663e-03, 0.1605606471164067e+01, 0.1700242913771285e+01, 0.9563440497488795e-01, 0.5924520849284005e-01, 0.1053295055699924e+01, 0.1140565764170461e+01, 0.1465159169633776e+00, 0.2036234219705987e+00, 0.1585124051340792e+01, 0.1431315653342504e+01, 0.4981502397231050e-01, 0.2448080089382756e-01, 0.7033973032949510e+00, 0.7941601570521848e+00, 0.1152436546510613e+00, 0.1446598374682775e+00, 0.9968711907896164e+00, 0.1297075908641373e+01, 0.4448645553200346e+00, 0.4841405038964208e+00, 0.1411591407973117e+01, 0.1017552654060970e+01, 0.9010174998427344e-01, 0.7455594793877945e-01, 0.8419881158177074e+00, 0.1026142136362350e+01, 0.2587099684834221e+00, 0.4554599945126984e+00, 0.1806527574243020e+01, 0.1455048625807853e+01, 0.1039810460775319e+00, 0.7731088667813635e-01, 0.1081842351702556e+01, 0.1021456859803743e+01, 0.1692539477932339e-01, 0.1363571444815687e-01, 0.8229228471769590e+00, 0.8447303166937503e+00, 0.3544318396494814e-01, 0.4231334180394086e-01, 0.1008469291521964e+01, 0.9661590126100381e+00, 0.3062892015595061e-05, 0.9665775484099522e-06, 0.3048973339666884e+00, 0.3049033804612119e+00, 0.7013072071892081e-05, 0.2341758973227438e-05, 0.1018113345824760e+00, 0.1018089928235306e+00, 0.2786425742647189e-13, 0.7379175991216932e-30, 0.1093987140067686e+00},
			i0:       1,
			n0:       16,
			pp:       1,
			tau:      0.10180650470263587,
			sigma:    0.10939871400676864,
			i0Out:    1,
			n0Out:    16,
			ppOut:    1,
			tauOut:   0.10180650470263587,
			sigmaOut: 0.10939871400676864,
			dminOut:  1.3142485785350155e-006,
			dmin1Out: 0.20309651253108729,
			dmin2Out: 0.57610166617362735,
			dnOut:    1.3142485785350155e-006,
			dnm1Out:  0.20309651253108729,
			dnm2Out:  0.81137144083676127,
		},
		{
			z:        []float64{0.2269772292606755e+01, 0.2371578794670729e+01, 0.1248859578385863e-08, 0.2638661645752538e-08, 0.9725961809002787e+00, 0.1074266900775131e+01, 0.2207509801202943e-03, 0.1357860766428602e-03, 0.1480145783712083e+01, 0.1581175077027170e+01, 0.1146359002276934e-02, 0.9979623676695663e-03, 0.1656535258559212e+01, 0.1700242913771285e+01, 0.4079180092843103e-01, 0.5924520849284005e-01, 0.1201590880509993e+01, 0.1140565764170461e+01, 0.2425529321011353e+00, 0.2036234219705987e+00, 0.1111437017432560e+01, 0.1431315653342504e+01, 0.1749237822536809e-01, 0.2448080089382756e-01, 0.8195211115924584e+00, 0.7941601570521848e+00, 0.2289566278084960e+00, 0.1446598374682775e+00, 0.1450453280026662e+01, 0.1297075908641373e+01, 0.3396444831847066e+00, 0.4841405038964208e+00, 0.6506576141124067e+00, 0.1017552654060970e+01, 0.1175810411452473e+00, 0.7455594793877945e-01, 0.1262214585027166e+01, 0.1026142136362350e+01, 0.5250426092262996e+00, 0.4554599945126984e+00, 0.9055103985570538e+00, 0.1455048625807853e+01, 0.8721019179982008e-01, 0.7731088667813635e-01, 0.8460758777494440e+00, 0.1021456859803743e+01, 0.1361402882064932e-01, 0.1363571444815687e-01, 0.7716231249744060e+00, 0.8447303166937503e+00, 0.5298106707064092e-01, 0.4231334180394086e-01, 0.8113724074143097e+00, 0.9661590126100381e+00, 0.3632274887524200e-06, 0.9665775484099522e-06, 0.2030988542900605e+00, 0.3049033804612119e+00, 0.1173872316183806e-05, 0.2341758973227438e-05, 0.1314248578535016e-05, 0.1018089928235306e+00, 0.1248859578385863e-08, 0.2638661645752538e-08},
			i0:       1,
			n0:       16,
			pp:       0,
			tau:      1.3142409824136984e-006,
			sigma:    0.21120521870940451,
			i0Out:    1,
			n0Out:    16,
			ppOut:    0,
			tauOut:   1.3142409824136984e-006,
			sigmaOut: 0.21120521870940451,
			dminOut:  8.4118841991824567e-018,
			dmin1Out: 0.20309744276965425,
			dmin2Out: 0.50018599148866161,
			dnOut:    8.4118841991824567e-018,
			dnm1Out:  0.20309744276965425,
			dnm2Out:  0.75834178007645303,
		},
		{
			z:        []float64{0.2269772292606755e+01, 0.2269770979614632e+01, 0.1248859578385863e-08, 0.5351359530665278e-09, 0.9725961809002787e+00, 0.9728156171042807e+00, 0.2207509801202943e-03, 0.3358741643642203e-03, 0.1480145783712083e+01, 0.1480954954309013e+01, 0.1146359002276934e-02, 0.1282269997958535e-02, 0.1656535258559212e+01, 0.1696043475248702e+01, 0.4079180092843103e-01, 0.2889964597634762e-01, 0.1201590880509993e+01, 0.1415242852393798e+01, 0.2425529321011353e+00, 0.1904848393814713e+00, 0.1111437017432560e+01, 0.9384432420354740e+00, 0.1749237822536809e-01, 0.1527569554079381e-01, 0.8195211115924584e+00, 0.1033200729619178e+01, 0.2289566278084960e+00, 0.3214195289148511e+00, 0.1450453280026662e+01, 0.1468676920055535e+01, 0.3396444831847066e+00, 0.1504703083827628e+00, 0.6506576141124067e+00, 0.6177670326339089e+00, 0.1175810411452473e+00, 0.2402402478867145e+00, 0.1262214585027166e+01, 0.1547015632125768e+01, 0.5250426092262996e+00, 0.3073217441808570e+00, 0.9055103985570538e+00, 0.6853975319350345e+00, 0.8721019179982008e-01, 0.1076549537133787e+00, 0.8460758777494440e+00, 0.7520336386157322e+00, 0.1361402882064932e-01, 0.1396865635082151e-01, 0.7716231249744060e+00, 0.8106342214532430e+00, 0.5298106707064092e-01, 0.5302931309687425e-01, 0.8113724074143097e+00, 0.7583421433039418e+00, 0.3632274887524200e-06, 0.9727942389020712e-07, 0.2030988542900605e+00, 0.2030986166419704e+00, 0.1173872316183806e-05, 0.7596112905317353e-11, 0.1314248578535016e-05, 0.8411884199182457e-17, 0.1248859578385863e-08, 0.5351359530665278e-09, 0.1093987140067686e+00},
			i0:       1,
			n0:       16,
			pp:       1,
			tau:      8.4118841988678429e-018,
			sigma:    0.21120653295038691,
			i0Out:    1,
			n0Out:    16,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.21120653295038691,
			dminOut:  8.4118841988678429e-018,
			dmin1Out: 0.20309858884628909,
			dmin2Out: 0.53283137411117854,
			dnOut:    8.4118841988678429e-018,
			dnm1Out:  0.20309858884628909,
			dnm2Out:  0.71080515929391042,
		},
		{
			z:        []float64{0.2269770980149768e+01, 0.2269770979614632e+01, 0.2293573303077261e-09, 0.5351359530665278e-09, 0.9731514910392876e+00, 0.9728156171042807e+00, 0.5111377954200868e-03, 0.3358741643642203e-03, 0.1481726086511552e+01, 0.1480954954309013e+01, 0.1467737987028945e-02, 0.1282269997958535e-02, 0.1723475383238021e+01, 0.1696043475248702e+01, 0.2373112943910766e-01, 0.2889964597634762e-01, 0.1581996562336162e+01, 0.1415242852393798e+01, 0.1129959536471923e+00, 0.1904848393814713e+00, 0.8407229839290754e+00, 0.9384432420354740e+00, 0.1877296098701645e-01, 0.1527569554079381e-01, 0.1335847297547013e+01, 0.1033200729619178e+01, 0.3533797947109679e+00, 0.3214195289148511e+00, 0.1265767433727330e+01, 0.1468676920055535e+01, 0.7343813202351114e-01, 0.1504703083827628e+00, 0.7845691484971122e+00, 0.6177670326339089e+00, 0.4737063898809231e+00, 0.2402402478867145e+00, 0.1380630986425702e+01, 0.1547015632125768e+01, 0.1525661578238559e+00, 0.3073217441808570e+00, 0.6404863278245572e+00, 0.6853975319350345e+00, 0.1264041760751794e+00, 0.1076549537133787e+00, 0.6395981188913744e+00, 0.7520336386157322e+00, 0.1770404028911661e-01, 0.1396865635082151e-01, 0.8459594942610007e+00, 0.8106342214532430e+00, 0.4753698401003136e-01, 0.5302931309687425e-01, 0.7108052565733343e+00, 0.7583421433039418e+00, 0.2779568135873871e-07, 0.9727942389020712e-07, 0.2030985888538852e+00, 0.2030986166419704e+00, 0.3146138162949754e-27, 0.7596112905317353e-11, 0.8411884198867843e-17, 0.8411884199182457e-17, 0.2293573303077261e-09, 0.5351359530665278e-09},
			i0:       1,
			n0:       16,
			pp:       0,
			tau:      8.4118841988678429e-018,
			sigma:    0.21120653295038691,
			i0Out:    1,
			n0Out:    16,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.21120653295038691,
			dminOut:  8.4118841988678429e-018,
			dmin1Out: 0.20309858045029386,
			dmin2Out: 0.51860540644834729,
			dnOut:    8.4118841988678429e-018,
			dnm1Out:  0.20309858045029386,
			dnm2Out:  0.67176795876146822,
		},
		{
			z:        []float64{0.2269770980149768e+01, 0.2269770980379126e+01, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.9731514910392876e+00, 0.9736626287363720e+00, 0.5111377954200868e-03, 0.7778528033461282e-03, 0.1481726086511552e+01, 0.1482415971695234e+01, 0.1467737987028945e-02, 0.1706410574351102e-02, 0.1723475383238021e+01, 0.1745500102102777e+01, 0.2373112943910766e-01, 0.2150819994097728e-01, 0.1581996562336162e+01, 0.1673484316042377e+01, 0.1129959536471923e+00, 0.5676676764251993e-01, 0.8407229839290754e+00, 0.8027291772735718e+00, 0.1877296098701645e-01, 0.3124068479314146e-01, 0.1335847297547013e+01, 0.1657986407464839e+01, 0.3533797947109679e+00, 0.2697830536296953e+00, 0.1265767433727330e+01, 0.1069422512121146e+01, 0.7343813202351114e-01, 0.5387701498318342e-01, 0.7845691484971122e+00, 0.1204398523394852e+01, 0.4737063898809231e+00, 0.5430210247136315e+00, 0.1380630986425702e+01, 0.9901761195359265e+00, 0.1525661578238559e+00, 0.9868601781741700e-01, 0.6404863278245572e+00, 0.6682044860823195e+00, 0.1264041760751794e+00, 0.1209927124430272e+00, 0.6395981188913744e+00, 0.5363094467374639e+00, 0.1770404028911661e-01, 0.2792585709699239e-01, 0.8459594942610007e+00, 0.8655706211740396e+00, 0.4753698401003136e-01, 0.3903729781186600e-01, 0.7108052565733343e+00, 0.6717679865571495e+00, 0.2779568135873871e-07, 0.8403591378512072e-08, 0.2030985888538852e+00, 0.2030985804502939e+00, 0.3146138162949754e-27, 0.1303059324279677e-43, 0.2112065329503869e+00},
			i0:       1,
			n0:       15,
			pp:       1,
			tau:      0.10154929022514693,
			sigma:    0.21120653295038691,
			i0Out:    1,
			n0Out:    15,
			ppOut:    1,
			tauOut:   0.10154929022514693,
			sigmaOut: 0.21120653295038691,
			dminOut:  0.10154928703201281,
			dmin1Out: 0.32391205918605420,
			dmin2Out: 0.32391205918605420,
			dnOut:    0.10154928703201281,
			dnm1Out:  0.53450854002684800,
			dnm2Out:  0.69531980574199759,
		},
		{
			z:        []float64{0.2168221690252314e+01, 0.2269770980379126e+01, 0.4415865678637858e-10, 0.9833566024906726e-10, 0.8728911912704126e+00, 0.9736626287363720e+00, 0.1321013925721922e-02, 0.7778528033461282e-03, 0.1381252078118717e+01, 0.1482415971695234e+01, 0.2156405683614187e-02, 0.1706410574351102e-02, 0.1663302606134993e+01, 0.1745500102102777e+01, 0.2163985983955575e-01, 0.2150819994097728e-01, 0.1607061933620194e+01, 0.1673484316042377e+01, 0.2835506195054301e-01, 0.5676676764251993e-01, 0.7040655098910235e+00, 0.8027291772735718e+00, 0.7356791380810462e-01, 0.3124068479314146e-01, 0.1752652257061283e+01, 0.1657986407464839e+01, 0.1646145547572216e+00, 0.2697830536296953e+00, 0.8571356821219610e+00, 0.1069422512121146e+01, 0.7570493055431493e-01, 0.5387701498318342e-01, 0.1570165327329021e+01, 0.1204398523394852e+01, 0.3424393862982660e+00, 0.5430210247136315e+00, 0.6448734608299306e+00, 0.9901761195359265e+00, 0.1022564019526126e+00, 0.9868601781741700e-01, 0.5853915063475871e+00, 0.6682044860823195e+00, 0.1108480973262629e+00, 0.1209927124430272e+00, 0.3518379162830466e+00, 0.5363094467374639e+00, 0.6870152520689508e-01, 0.2792585709699239e-01, 0.7343571035538636e+00, 0.8655706211740396e+00, 0.3571015630515456e-01, 0.3903729781186600e-01, 0.5345085484304394e+00, 0.6717679865571495e+00, 0.3193134112956561e-08, 0.8403591378512072e-08, 0.1015492870320128e+00, 0.2030985804502939e+00, 0.4415865678637858e-10, 0.9833566024906726e-10},
			i0:       1,
			n0:       15,
			pp:       0,
			tau:      0.10154100618138810,
			sigma:    0.31275582317553385,
			i0Out:    1,
			n0Out:    15,
			ppOut:    0,
			tauOut:   0.10154100618138810,
			sigmaOut: 0.31275582317553385,
			dminOut:  8.2800215414347011e-006,
			dmin1Out: 0.16866357962181588,
			dmin2Out: 0.16866357962181588,
			dnOut:    8.2800215414347011e-006,
			dnm1Out:  0.39110725789187228,
			dnm2Out:  0.42026820370851276,
		},
		{
			z:        []float64{0.2168221690252314e+01, 0.2066680684115085e+01, 0.4415865678637858e-10, 0.1865101988102620e-10, 0.8728911912704126e+00, 0.7726711989960955e+00, 0.1321013925721922e-02, 0.2361487308570419e-02, 0.1381252078118717e+01, 0.1279505990312372e+01, 0.2156405683614187e-02, 0.2803234389363145e-02, 0.1663302606134993e+01, 0.1580598225403798e+01, 0.2163985983955575e-01, 0.2200217261925751e-01, 0.1607061933620194e+01, 0.1511873816770092e+01, 0.2835506195054301e-01, 0.1320468740761088e-01, 0.7040655098910235e+00, 0.6628877301101292e+00, 0.7356791380810462e-01, 0.1945110224949301e+00, 0.1752652257061283e+01, 0.1621214783142186e+01, 0.1646145547572216e+00, 0.8703165684534678e-01, 0.8571356821219610e+00, 0.7442679496495411e+00, 0.7570493055431493e-01, 0.1597129865933493e+00, 0.1570165327329021e+01, 0.1651350720852550e+01, 0.3424393862982660e+00, 0.1337269360034139e+00, 0.6448734608299306e+00, 0.5118619205977412e+00, 0.1022564019526126e+00, 0.1169456581236225e+00, 0.5853915063475871e+00, 0.4777529393688394e+00, 0.1108480973262629e+00, 0.8163333047984263e-01, 0.3518379162830466e+00, 0.2373651048287110e+00, 0.6870152520689508e-01, 0.2125478936639627e+00, 0.7343571035538636e+00, 0.4559783600136673e+00, 0.3571015630515456e-01, 0.4186028435717903e-01, 0.5345085484304394e+00, 0.3911072610850064e+00, 0.3193134112956561e-08, 0.8290832843879624e-09, 0.1015492870320128e+00, 0.8280021541434701e-05, 0.4415865678637858e-10, 0.1865101988102620e-10, 0.2112065329503869e+00},
			i0:       1,
			n0:       15,
			pp:       1,
			tau:      8.2795951083136037e-006,
			sigma:    0.41429682935692197,
			i0Out:    1,
			n0Out:    15,
			ppOut:    1,
			tauOut:   8.2795951083136037e-006,
			sigmaOut: 0.41429682935692197,
			dminOut:  4.2641220534843968e-010,
			dmin1Out: 0.19554701532405336,
			dmin2Out: 0.19554701532405336,
			dnOut:    4.2641220534843968e-010,
			dnm1Out:  0.32821333105682254,
			dnm2Out:  0.21848306981706411,
		},
		{
			z:        []float64{0.2066672404538628e+01, 0.2066680684115085e+01, 0.6973096395115281e-11, 0.1865101988102620e-10, 0.7750244067025847e+00, 0.7726711989960955e+00, 0.3898634844569491e-02, 0.2361487308570419e-02, 0.1278402310262057e+01, 0.1279505990312372e+01, 0.3465878671879142e-02, 0.2803234389363145e-02, 0.1599126239756068e+01, 0.1580598225403798e+01, 0.2080167773382635e-01, 0.2200217261925751e-01, 0.1504268546848768e+01, 0.1511873816770092e+01, 0.5818924606767697e-02, 0.1320468740761088e-01, 0.8515715484031834e+00, 0.6628877301101292e+00, 0.3703084558710277e+00, 0.1945110224949301e+00, 0.1337929704521397e+01, 0.1621214783142186e+01, 0.4841425717359337e-01, 0.8703165684534678e-01, 0.8555583994741887e+00, 0.7442679496495411e+00, 0.3082690272254148e+00, 0.1597129865933493e+00, 0.1476800350035441e+01, 0.1651350720852550e+01, 0.4635002036444261e-01, 0.1337269360034139e+00, 0.5824492787618127e+00, 0.5118619205977412e+00, 0.9592445892242578e-01, 0.1169456581236225e+00, 0.4634535313311479e+00, 0.4777529393688394e+00, 0.4180980990954932e-01, 0.8163333047984263e-01, 0.4080949089880160e+00, 0.2373651048287110e+00, 0.2374870106014949e+00, 0.2125478936639627e+00, 0.2603433541742431e+00, 0.4559783600136673e+00, 0.6288565043307552e-01, 0.4186028435717903e-01, 0.3282133318859058e+00, 0.3911072610850064e+00, 0.2091574834858362e-13, 0.8290832843879624e-09, 0.4264122053484397e-09, 0.8280021541434701e-05, 0.6973096395115281e-11, 0.1865101988102620e-10},
			i0:       1,
			n0:       15,
			pp:       0,
			tau:      4.2641207498271701e-010,
			sigma:    0.41430510895203027,
			i0Out:    1,
			n0Out:    15,
			ppOut:    0,
			tauOut:   4.2641207498271701e-010,
			sigmaOut: 0.41430510895203027,
			dminOut:  1.3036568472020817e-016,
			dmin1Out: 0.15841174212052453,
			dmin2Out: 0.15841174212052453,
			dnOut:    1.3036568472020817e-016,
			dnm1Out:  0.23494558610432464,
			dnm2Out:  0.15841174212052453,
		},
		{
			z:        []float64{0.2066672404538628e+01, 0.2066672404119188e+01, 0.6973096395115281e-11, 0.2614986238618434e-11, 0.7750244067025847e+00, 0.7789230411181270e+00, 0.3898634844569491e-02, 0.6398608757305904e-02, 0.1278402310262057e+01, 0.1275469579750218e+01, 0.3465878671879142e-02, 0.4345362379476143e-02, 0.1599126239756068e+01, 0.1615582554684006e+01, 0.2080167773382635e-01, 0.1936843737632438e-01, 0.1504268546848768e+01, 0.1490719033652799e+01, 0.5818924606767697e-02, 0.3324054047451484e-02, 0.8515715484031834e+00, 0.1218555949800348e+01, 0.3703084558710277e+00, 0.4065850919906259e+00, 0.1337929704521397e+01, 0.9797588692779520e+00, 0.4841425717359337e-01, 0.4227695780870786e-01, 0.8555583994741887e+00, 0.1121550468464483e+01, 0.3082690272254148e+00, 0.4059129037098644e+00, 0.1476800350035441e+01, 0.1117237466263607e+01, 0.4635002036444261e-01, 0.2416365074306882e-01, 0.5824492787618127e+00, 0.6542100865147574e+00, 0.9592445892242578e-01, 0.6795451513972497e-01, 0.4634535313311479e+00, 0.4373088256745602e+00, 0.4180980990954932e-01, 0.3901675330591518e-01, 0.4080949089880160e+00, 0.6065651658571838e+00, 0.2374870106014949e+00, 0.1019316116273065e+00, 0.2603433541742431e+00, 0.2212973925536000e+00, 0.6288565043307552e-01, 0.9326774535516916e-01, 0.3282133318859058e+00, 0.2349455861043456e+00, 0.2091574834858362e-13, 0.3796083394336032e-22, 0.4264122053484397e-09, 0.1303656847202082e-15, 0.6973096395115281e-11, 0.2614986238618434e-11, 0.2112065329503869e+00},
			i0:       1,
			n0:       15,
			pp:       1,
			tau:      1.3036568471812905e-016,
			sigma:    0.41430510937844234,
			i0Out:    1,
			n0Out:    15,
			ppOut:    1,
			tauOut:   1.3036568471812905e-016,
			sigmaOut: 0.41430510937844234,
			dminOut:  2.0791168714198411e-027,
			dmin1Out: 0.15670572686712736,
			dmin2Out: 0.18680490956440032,
			dnOut:    2.0791168714198411e-027,
			dnm1Out:  0.15670572686712736,
			dnm2Out:  0.18680490956440032,
		},
		{
			z:        []float64{0.2066672404121803e+01, 0.2066672404119188e+01, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.7853216498744473e+00, 0.7789230411181270e+00, 0.1039221422709001e-01, 0.6398608757305904e-02, 0.1269422727902604e+01, 0.1275469579750218e+01, 0.5530302475095172e-02, 0.4345362379476143e-02, 0.1629420689585235e+01, 0.1615582554684006e+01, 0.1771973219288664e-01, 0.1936843737632438e-01, 0.1476323355507364e+01, 0.1490719033652799e+01, 0.2743671176012720e-02, 0.3324054047451484e-02, 0.1622397370614960e+01, 0.1218555949800348e+01, 0.2455350071499520e+00, 0.4065850919906259e+00, 0.7765008199367077e+00, 0.9797588692779520e+00, 0.6106335063429082e-01, 0.4227695780870786e-01, 0.1466400021540057e+01, 0.1121550468464483e+01, 0.3092615230516922e+00, 0.4059129037098644e+00, 0.8321395939549830e+00, 0.1117237466263607e+01, 0.1899693772291602e-01, 0.2416365074306882e-01, 0.7031676639315663e+00, 0.6542100865147574e+00, 0.4226176876348708e-01, 0.6795451513972497e-01, 0.4340638102169883e+00, 0.4373088256745602e+00, 0.5452240634477346e-01, 0.3901675330591518e-01, 0.6539743711397168e+00, 0.6065651658571838e+00, 0.3449248298919955e-01, 0.1019316116273065e+00, 0.2800726549195695e+00, 0.2212973925536000e+00, 0.7823985923721809e-01, 0.9326774535516916e-01, 0.1567057268671274e+00, 0.2349455861043456e+00, 0.3158014839988858e-37, 0.3796083394336032e-22},
			i0:       1,
			n0:       14,
			pp:       0,
			tau:      5.2226904068357879e-002,
			sigma:    0.41430510937844245,
			i0Out:    1,
			n0Out:    14,
			ppOut:    0,
			tauOut:   5.2226904068357879e-002,
			sigmaOut: 0.41430510937844245,
			dminOut:  6.1979733096699005e-002,
			dmin1Out: 0.21025182606004778,
			dmin2Out: 0.35454858327394784,
			dnOut:    6.1979733096699005e-002,
			dnm1Out:  0.21025182606004778,
			dnm2Out:  0.51458348500658069,
		},
		{
			z:        []float64{0.2066672404121803e+01, 0.2014445500054431e+01, 0.9855809897129084e-12, 0.3842238913414725e-12, 0.7853216498744473e+00, 0.7434869600327952e+00, 0.1039221422709001e-01, 0.1774357001838869e-01, 0.1269422727902604e+01, 0.1204982556290953e+01, 0.5530302475095172e-02, 0.7478273627729330e-02, 0.1629420689585235e+01, 0.1587435244082034e+01, 0.1771973219288664e-01, 0.1647944669694028e-01, 0.1476323355507364e+01, 0.1410360675918078e+01, 0.2743671176012720e-02, 0.3156160674217245e-02, 0.1622397370614960e+01, 0.1812549313022337e+01, 0.2455350071499520e+00, 0.1051878329628395e+00, 0.7765008199367077e+00, 0.6801494335398012e+00, 0.6106335063429082e-01, 0.1316523902981273e+00, 0.1466400021540057e+01, 0.1591782250225264e+01, 0.3092615230516922e+00, 0.1616733433116970e+00, 0.8321395939549830e+00, 0.6372362842978442e+00, 0.1899693772291602e-01, 0.2096244776016672e-01, 0.7031676639315663e+00, 0.6722400808665288e+00, 0.4226176876348708e-01, 0.2728832287468248e-01, 0.4340638102169883e+00, 0.4090709896187213e+00, 0.5452240634477346e-01, 0.8716398206477828e-01, 0.6539743711397168e+00, 0.5490759679957803e+00, 0.3449248298919955e-01, 0.1759392479116377e-01, 0.2800726549195695e+00, 0.2884916852972659e+00, 0.7823985923721809e-01, 0.4249908970207047e-01, 0.1567057268671274e+00, 0.6197973309669901e-01, 0.3158014839988858e-37, 0.3842238913414725e-12, 0.4143051093784424e+00},
			i0:       1,
			n0:       14,
			pp:       1,
			tau:      4.3992746597899146e-002,
			sigma:    0.46653201344680034,
			i0Out:    1,
			n0Out:    14,
			ppOut:    1,
			tauOut:   4.3992746597899146e-002,
			sigmaOut: 0.46653201344680034,
			dminOut:  8.3978432207528000e-003,
			dmin1Out: 0.23219513064786862,
			dmin2Out: 0.34734604130905122,
			dnOut:    8.3978432207528000e-003,
			dnm1Out:  0.23219513064786862,
			dnm2Out:  0.39493697562207530,
		},
		{
			z:        []float64{0.1970452753456916e+01, 0.2014445500054431e+01, 0.1449745254963753e-12, 0.3842238913414725e-12, 0.7172377834531399e+00, 0.7434869600327952e+00, 0.2980976860358389e-01, 0.1774357001838869e-01, 0.1138658314717199e+01, 0.1204982556290953e+01, 0.1042566937606312e-01, 0.7478273627729330e-02, 0.1549496274805013e+01, 0.1587435244082034e+01, 0.1499968987352189e-01, 0.1647944669694028e-01, 0.1354524400120875e+01, 0.1410360675918078e+01, 0.4223398900256122e-02, 0.3156160674217245e-02, 0.1869521000487022e+01, 0.1812549313022337e+01, 0.3826832915293116e-01, 0.1051878329628395e+00, 0.7295407480870981e+00, 0.6801494335398012e+00, 0.2872518617030954e+00, 0.1316523902981273e+00, 0.1422210985235967e+01, 0.1591782250225264e+01, 0.7243940711431240e-01, 0.1616733433116970e+00, 0.5417665783457993e+00, 0.6372362842978442e+00, 0.2601082853889214e-01, 0.2096244776016672e-01, 0.6295248286044199e+00, 0.6722400808665288e+00, 0.1773220171177086e-01, 0.2728832287468248e-01, 0.4345100233738295e+00, 0.4090709896187213e+00, 0.1101462457758059e+00, 0.8716398206477828e-01, 0.4125309004132391e+00, 0.5490759679957803e+00, 0.1230380805149813e-01, 0.1759392479116377e-01, 0.2746942203499391e+00, 0.2884916852972659e+00, 0.9589143278047064e-02, 0.4249908970207047e-01, 0.8397843220752800e-02, 0.6197973309669901e-01, 0.1449745254963753e-12, 0.3842238913414725e-12},
			i0:       1,
			n0:       14,
			pp:       0,
			tau:      8.0121275381568099e-003,
			sigma:    0.51052476004469949,
			i0Out:    1,
			n0Out:    14,
			ppOut:    0,
			tauOut:   8.0121275381568099e-003,
			sigmaOut: 0.51052476004469949,
			dminOut:  8.3015533671799979e-005,
			dmin1Out: 0.25644350373773794,
			dmin2Out: 0.31779880038403513,
			dnOut:    8.3015533671799979e-005,
			dnm1Out:  0.25644350373773794,
			dnm2Out:  0.31779880038403513,
		},
		{
			z:        []float64{0.1970452753456916e+01, 0.1962440625918904e+01, 0.1449745254963753e-12, 0.5298565773193886e-13, 0.7172377834531399e+00, 0.7390354245185139e+00, 0.2980976860358389e-01, 0.4592897681783073e-01, 0.1138658314717199e+01, 0.1095142879737275e+01, 0.1042566937606312e-01, 0.1475107600976594e-01, 0.1549496274805013e+01, 0.1541732761130612e+01, 0.1499968987352189e-01, 0.1317831886314191e-01, 0.1354524400120875e+01, 0.1337557352619832e+01, 0.4223398900256122e-02, 0.5903098601340335e-02, 0.1869521000487022e+01, 0.1893874103500456e+01, 0.3826832915293116e-01, 0.1474137347708128e-01, 0.7295407480870981e+00, 0.9940391087749554e+00, 0.2872518617030954e+00, 0.4109825756725979e+00, 0.1422210985235967e+01, 0.1075655689139524e+01, 0.7243940711431240e-01, 0.3648495529374628e-01, 0.5417665783457993e+00, 0.5232803240527883e+00, 0.2601082853889214e-01, 0.3129195122603765e-01, 0.6295248286044199e+00, 0.6079529515519964e+00, 0.1773220171177086e-01, 0.1267338099203561e-01, 0.4345100233738295e+00, 0.5239707606194429e+00, 0.1101462457758059e+00, 0.8671997249104711e-01, 0.4125309004132391e+00, 0.3301026084355332e+00, 0.1230380805149813e-01, 0.1023858907404432e-01, 0.2746942203499391e+00, 0.2660326470157850e+00, 0.9589143278047064e-02, 0.3027001489241909e-03, 0.8397843220752800e-02, 0.8301553367179998e-04, 0.1449745254963753e-12, 0.5298565773193886e-13, 0.4143051093784424e+00},
			i0:       1,
			n0:       14,
			pp:       1,
			tau:      8.2898056486573887e-005,
			sigma:    0.51853688758285632,
			i0Out:    1,
			n0Out:    14,
			ppOut:    1,
			tauOut:   8.2898056486573887e-005,
			sigmaOut: 0.51853688758285632,
			dminOut:  1.9676836973621424e-008,
			dmin1Out: 0.25663722751477031,
			dmin2Out: 0.28224921557690585,
			dnOut:    1.9676836973621424e-008,
			dnm1Out:  0.25663722751477031,
			dnm2Out:  0.28224921557690585,
		},
		{
			z:        []float64{0.1962357727862470e+01, 0.1962440625918904e+01, 0.1995470932711637e-13, 0.5298565773193886e-13, 0.7848815032798381e+00, 0.7390354245185139e+00, 0.6408456783027588e-01, 0.4592897681783073e-01, 0.1045726489860278e+01, 0.1095142879737275e+01, 0.2174776805091989e-01, 0.1475107600976594e-01, 0.1533080413886347e+01, 0.1541732761130612e+01, 0.1149760777771624e-01, 0.1317831886314191e-01, 0.1331879945386970e+01, 0.1337557352619832e+01, 0.8393943921304423e-02, 0.5903098601340335e-02, 0.1900138634999746e+01, 0.1893874103500456e+01, 0.7711806645770664e-02, 0.1474137347708128e-01, 0.1397226979745296e+01, 0.9940391087749554e+00, 0.3163950825942626e+00, 0.4109825756725979e+00, 0.7956626637825215e+00, 0.1075655689139524e+01, 0.2399491656225486e-01, 0.3648495529374628e-01, 0.5304944606600845e+00, 0.5232803240527883e+00, 0.3586094769777509e-01, 0.3129195122603765e-01, 0.5846824867897703e+00, 0.6079529515519964e+00, 0.1135741402906866e-01, 0.1267338099203561e-01, 0.5992504210249349e+00, 0.5239707606194429e+00, 0.4777049480214081e-01, 0.8671997249104711e-01, 0.2924878046509502e+00, 0.3301026084355332e+00, 0.9312521444528056e-02, 0.1023858907404432e-01, 0.2569399276636945e+00, 0.2660326470157850e+00, 0.9780034825247492e-07, 0.3027001489241909e-03, 0.1967683697362142e-07, 0.8301553367179998e-04, 0.1995470932711637e-13, 0.5298565773193886e-13},
			i0:       1,
			n0:       14,
			pp:       0,
			tau:      1.9676827722764296e-008,
			sigma:    0.51861978563934286,
			i0Out:    1,
			n0Out:    14,
			ppOut:    0,
			tauOut:   1.9676827722764296e-008,
			sigmaOut: 0.51861978563934286,
			dminOut:  1.5032959860012969e-015,
			dmin1Out: 0.24838793989138258,
			dmin2Out: 0.27047787997045275,
			dnOut:    1.5032959860012969e-015,
			dnm1Out:  0.24838793989138258,
			dnm2Out:  0.27047787997045275,
		},
		{
			z:        []float64{0.1962357727862470e+01, 0.1962357708185662e+01, 0.1995470932711637e-13, 0.7981257539768321e-14, 0.7848815032798381e+00, 0.8489660514332782e+00, 0.6408456783027588e-01, 0.7893711422056097e-01, 0.1045726489860278e+01, 0.9885371240138099e+00, 0.2174776805091989e-01, 0.3372769361380378e-01, 0.1533080413886347e+01, 0.1510850308373431e+01, 0.1149760777771624e-01, 0.1013563894066502e-01, 0.1331879945386970e+01, 0.1330138230690781e+01, 0.8393943921304423e-02, 0.1199097716077874e-01, 0.1900138634999746e+01, 0.1895859444807910e+01, 0.7711806645770664e-02, 0.5683514322519618e-02, 0.1397226979745296e+01, 0.1707938528340212e+01, 0.3163950825942626e+00, 0.1473962616612956e+00, 0.7956626637825215e+00, 0.6722612990066530e+00, 0.2399491656225486e-01, 0.1893485515094503e-01, 0.5304944606600845e+00, 0.5474205335300870e+00, 0.3586094769777509e-01, 0.3830193935796279e-01, 0.5846824867897703e+00, 0.5577379417840483e+00, 0.1135741402906866e-01, 0.1220274725600271e-01, 0.5992504210249349e+00, 0.6348181488942451e+00, 0.4777049480214081e-01, 0.2200990500366971e-01, 0.2924878046509502e+00, 0.2797904014149808e+00, 0.9312521444528056e-02, 0.8551968095484232e-02, 0.2569399276636945e+00, 0.2483880376917308e+00, 0.9780034825247492e-07, 0.7747561140265878e-14, 0.1967683697362142e-07, 0.1503295986001297e-14, 0.1995470932711637e-13, 0.7981257539768321e-14, 0.4143051093784424e+00},
			i0:       1,
			n0:       14,
			pp:       1,
			tau:      1.5032959860012393e-015,
			sigma:    0.51861980531617058,
			i0Out:    1,
			n0Out:    14,
			ppOut:    1,
			tauOut:   1.5032959860012393e-015,
			sigmaOut: 0.51861980531617058,
			dminOut:  9.2691156363468887e-030,
			dmin1Out: 0.24076768543540897,
			dmin2Out: 0.27020241256687788,
			dnOut:    9.2691156363468887e-030,
			dnm1Out:  0.24076768543540897,
			dnm2Out:  0.27020241256687788,
		},
		{
			z:        []float64{0.1962357708185669e+01, 0.1962357708185662e+01, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.9279031656538340e+00, 0.8489660514332782e+00, 0.8409527066820451e-01, 0.7893711422056097e-01, 0.9381695469594078e+00, 0.9885371240138099e+00, 0.5431587122209677e-01, 0.3372769361380378e-01, 0.1466670076091998e+01, 0.1510850308373431e+01, 0.9192115573380727e-02, 0.1013563894066502e-01, 0.1332937092278178e+01, 0.1330138230690781e+01, 0.1705497388769041e-01, 0.1199097716077874e-01, 0.1884487985242738e+01, 0.1895859444807910e+01, 0.5151050664063701e-02, 0.5683514322519618e-02, 0.1850183739337442e+01, 0.1707938528340212e+01, 0.5355619565040129e-01, 0.1473962616612956e+00, 0.6376399585071952e+00, 0.6722612990066530e+00, 0.1625577000116482e-01, 0.1893485515094503e-01, 0.5694667028868834e+00, 0.5474205335300870e+00, 0.3751307094084999e-01, 0.3830193935796279e-01, 0.5324276180991995e+00, 0.5577379417840483e+00, 0.1454944326918194e-01, 0.1220274725600271e-01, 0.6422786106287314e+00, 0.6348181488942451e+00, 0.9587988848101409e-02, 0.2200990500366971e-01, 0.2787543806623621e+00, 0.2797904014149808e+00, 0.7620352256320365e-02, 0.8551968095484232e-02, 0.2407676854354167e+00, 0.2483880376917308e+00, 0.4837392336267432e-28, 0.7747561140265878e-14},
			i0:       1,
			n0:       13,
			pp:       0,
			tau:      0.19361025134591583,
			sigma:    0.51861980531617213,
			i0Out:    1,
			n0Out:    13,
			ppOut:    0,
			tauOut:   0.19361025134591583,
			sigmaOut: 0.51861980531617213,
			dminOut:  2.5949209826162500e-002,
			dmin1Out: 7.8890170839591350e-002,
			dmin2Out: 0.28791033268116917,
			dnOut:    2.5949209826162500e-002,
			dnm1Out:  7.8890170839591350e-002,
			dnm2Out:  0.41777236268627749,
		},
		{
			z:        []float64{0.1962357708185669e+01, 0.1768747456839757e+01, 0.3452895805257589e-14, 0.1811424766142649e-14, 0.9279031656538340e+00, 0.8183881849761209e+00, 0.8409527066820451e-01, 0.9640366690596863e-01, 0.9381695469594078e+00, 0.7024714999296202e+00, 0.5431587122209677e-01, 0.1134045480653624e+00, 0.1466670076091998e+01, 0.1168847392254101e+01, 0.9192115573380727e-02, 0.1048255904531585e-01, 0.1332937092278178e+01, 0.1145899255774637e+01, 0.1705497388769041e-01, 0.2804774784346500e-01, 0.1884487985242738e+01, 0.1667981036717421e+01, 0.5151050664063701e-02, 0.5713728135608638e-02, 0.1850183739337442e+01, 0.1704415955506319e+01, 0.5355619565040129e-01, 0.2003593680404179e-01, 0.6376399585071952e+00, 0.4402495403584025e+00, 0.1625577000116482e-01, 0.2102698332839761e-01, 0.5694667028868834e+00, 0.3923425391534199e+00, 0.3751307094084999e-01, 0.5090703407211448e-01, 0.5324276180991995e+00, 0.3024597759503511e+00, 0.1454944326918194e-01, 0.3089599659653809e-01, 0.6422786106287314e+00, 0.4273603515343789e+00, 0.9587988848101409e-02, 0.6253958476854957e-02, 0.2787543806623621e+00, 0.8651052309591171e-01, 0.7620352256320365e-02, 0.2120822426333837e-01, 0.2407676854354167e+00, 0.2594920982616250e-01, 0.4837392336267432e-28, 0.1811424766142649e-14, 0.5186198053161721e+00},
			i0:       1,
			n0:       13,
			pp:       1,
			tau:      1.8780556192507153e-002,
			sigma:    0.71223005666208794,
			i0Out:    1,
			n0Out:    13,
			ppOut:    1,
			tauOut:   1.8780556192507153e-002,
			sigmaOut: 0.71223005666208794,
			dminOut:  8.7653100724300811e-004,
			dmin1Out: 6.6256164803006098e-002,
			dmin2Out: 0.24571773655185866,
			dnOut:    8.7653100724300811e-004,
			dnm1Out:  6.6256164803006098e-002,
			dnm2Out:  0.36084635920253871,
		},
		{
			z:        []float64{0.1749966900647251e+01, 0.1768747456839757e+01, 0.8471295234418270e-15, 0.1811424766142649e-14, 0.8960112956895816e+00, 0.8183881849761209e+00, 0.7558032897122403e-01, 0.9640366690596863e-01, 0.7215151628312513e+00, 0.7024714999296202e+00, 0.1837142406762631e+00, 0.1134045480653624e+00, 0.9768351544306461e+00, 0.1168847392254101e+01, 0.1229681032071614e-01, 0.1048255904531585e-01, 0.1142869637104878e+01, 0.1145899255774637e+01, 0.4093477506677201e-01, 0.2804774784346500e-01, 0.1613979433593750e+01, 0.1667981036717421e+01, 0.6033886923870183e-02, 0.5713728135608638e-02, 0.1699637449193983e+01, 0.1704415955506319e+01, 0.5189819730562233e-02, 0.2003593680404179e-01, 0.4373061477637308e+00, 0.4402495403584025e+00, 0.1886499897608887e-01, 0.2102698332839761e-01, 0.4056040180569384e+00, 0.3923425391534199e+00, 0.3796148320598534e-01, 0.5090703407211448e-01, 0.2766137331483968e+00, 0.3024597759503511e+00, 0.4773343613933298e-01, 0.3089599659653809e-01, 0.3671003176793937e+00, 0.4273603515343789e+00, 0.1473802100398464e-02, 0.6253958476854957e-02, 0.8746438906634448e-01, 0.8651052309591171e-01, 0.6292122626412339e-02, 0.2120822426333837e-01, 0.8765310072430081e-03, 0.2594920982616250e-01, 0.8471295234418270e-15, 0.1811424766142649e-14},
			i0:       1,
			n0:       13,
			pp:       0,
			tau:      8.1622622999092049e-004,
			sigma:    0.73101061285459512,
			i0Out:    1,
			n0Out:    13,
			ppOut:    0,
			tauOut:   8.1622622999092049e-004,
			sigmaOut: 0.73101061285459512,
			dminOut:  6.9556311345050133e-007,
			dmin1Out: 8.6231166987816729e-002,
			dmin2Out: 0.25114202935985525,
			dnOut:    6.9556311345050133e-007,
			dnm1Out:  8.6231166987816729e-002,
			dnm2Out:  0.30765445605457048,
		},
		{
			z:        []float64{0.1749966900647251e+01, 0.1749150674417261e+01, 0.8471295234418270e-15, 0.4339463906783712e-15, 0.8960112956895816e+00, 0.9707753984308143e+00, 0.7558032897122403e-01, 0.5617401661873561e-01, 0.7215151628312513e+00, 0.8482391606587878e+00, 0.1837142406762631e+00, 0.2115659556707206e+00, 0.9768351544306461e+00, 0.7767497828506508e+00, 0.1229681032071614e-01, 0.1809289356632693e-01, 0.1142869637104878e+01, 0.1164895292375332e+01, 0.4093477506677201e-01, 0.5671572845129930e-01, 0.1613979433593750e+01, 0.1562481365836330e+01, 0.6033886923870183e-02, 0.6563547191183529e-02, 0.1699637449193983e+01, 0.1697447495503371e+01, 0.5189819730562233e-02, 0.1337031089310571e-02, 0.4373061477637308e+00, 0.4540178894205181e+00, 0.1886499897608887e-01, 0.1685334336738995e-01, 0.4056040180569384e+00, 0.4258959316655428e+00, 0.3796148320598534e-01, 0.2465547755855056e-01, 0.2766137331483968e+00, 0.2988754654991882e+00, 0.4773343613933298e-01, 0.5862963539483231e-01, 0.3671003176793937e+00, 0.3091282581549689e+00, 0.1473802100398464e-02, 0.4169958485368410e-03, 0.8746438906634448e-01, 0.9252328961422907e-01, 0.6292122626412339e-02, 0.5960921413863723e-04, 0.8765310072430081e-03, 0.6955631134505013e-06, 0.8471295234418270e-15, 0.4339463906783712e-15, 0.5186198053161721e+00},
			i0:       1,
			n0:       13,
			pp:       1,
			tau:      6.9511331676175615e-007,
			sigma:    0.73182683908458601,
			i0Out:    1,
			n0Out:    13,
			ppOut:    1,
			tauOut:   6.9511331676175615e-007,
			sigmaOut: 0.73182683908458601,
			dminOut:  1.2278907249223888e-012,
			dmin1Out: 9.2372077429255559e-002,
			dmin2Out: 0.25591158693238830,
			dnOut:    1.2278907249223888e-012,
			dnm1Out:  9.2372077429255559e-002,
			dnm2Out:  0.25591158693238830,
		},
		{
			z:        []float64{0.1749149979303945e+01, 0.1749150674417261e+01, 0.2408395422307052e-15, 0.4339463906783712e-15, 0.1026948719936233e+01, 0.9707753984308143e+00, 0.4639861736277134e-01, 0.5617401661873561e-01, 0.1013405803853420e+01, 0.8482391606587878e+00, 0.1621599259654447e+00, 0.2115659556707206e+00, 0.6326820553382161e+00, 0.7767497828506508e+00, 0.3331266686486833e-01, 0.1809289356632693e-01, 0.1188297658848447e+01, 0.1164895292375332e+01, 0.7457497555020480e-01, 0.5671572845129930e-01, 0.1494469242363992e+01, 0.1562481365836330e+01, 0.7455005714048079e-02, 0.6563547191183529e-02, 0.1691328825765317e+01, 0.1697447495503371e+01, 0.3589107121045615e-03, 0.1337031089310571e-02, 0.4705116269624868e+00, 0.4540178894205181e+00, 0.1525524548983380e-01, 0.1685334336738995e-01, 0.4352954686209429e+00, 0.4258959316655428e+00, 0.1692854133253905e-01, 0.2465547755855056e-01, 0.3405758644481647e+00, 0.2988754654991882e+00, 0.5321597610926388e-01, 0.5862963539483231e-01, 0.2563285827809251e+00, 0.3091282581549689e+00, 0.1505170716567330e-03, 0.4169958485368410e-03, 0.9243168664339420e-01, 0.9252328961422907e-01, 0.4485687980202113e-09, 0.5960921413863723e-04, 0.1227890724922389e-11, 0.6955631134505013e-06, 0.2408395422307052e-15, 0.4339463906783712e-15},
			i0:       1,
			n0:       13,
			pp:       0,
			tau:      1.2278907189544363e-012,
			sigma:    0.73182753419790281,
			i0Out:    1,
			n0Out:    13,
			ppOut:    0,
			tauOut:   1.2278907189544363e-012,
			sigmaOut: 0.73182753419790281,
			dminOut:  4.9598525010503808e-024,
			dmin1Out: 9.2368632133320736e-002,
			dmin2Out: 0.22049265431403467,
			dnOut:    4.9598525010503808e-024,
			dnm1Out:  9.2368632133320736e-002,
			dnm2Out:  0.22049265431403467,
		},
		{
			z:        []float64{0.1749149979303945e+01, 0.1749149979302717e+01, 0.2408395422307052e-15, 0.1414000300319855e-15, 0.1026948719936233e+01, 0.1073347337297776e+01, 0.4639861736277134e-01, 0.4380746706334982e-01, 0.1013405803853420e+01, 0.1131758262754287e+01, 0.1621599259654447e+00, 0.9065158049178278e-01, 0.6326820553382161e+00, 0.5753431417100738e+00, 0.3331266686486833e-01, 0.6880305191066147e-01, 0.1188297658848447e+01, 0.1194069582486762e+01, 0.7457497555020480e-01, 0.9333627524262254e-01, 0.1494469242363992e+01, 0.1408587972834190e+01, 0.7455005714048079e-02, 0.8951422490882578e-02, 0.1691328825765317e+01, 0.1682736313985311e+01, 0.3589107121045615e-03, 0.1003553923945662e-03, 0.4705116269624868e+00, 0.4856665170586981e+00, 0.1525524548983380e-01, 0.1367304312976171e-01, 0.4352954686209429e+00, 0.4385509668224923e+00, 0.1692854133253905e-01, 0.1314659648329904e-01, 0.3405758644481647e+00, 0.3806452440729016e+00, 0.5321597610926388e-01, 0.3583592846566260e-01, 0.2563285827809251e+00, 0.2206431713856914e+00, 0.1505170716567330e-03, 0.6305450884558253e-04, 0.9243168664339420e-01, 0.9236863258188953e-01, 0.4485687980202113e-09, 0.5962992535266723e-20, 0.1227890724922389e-11, 0.4959852501050381e-23, 0.2408395422307052e-15, 0.1414000300319855e-15, 0.5186198053161721e+00},
			i0:       1,
			n0:       13,
			pp:       1,
			tau:      4.9598525010503808e-024,
			sigma:    0.73182753419913071,
			i0Out:    1,
			n0Out:    13,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.73182753419913071,
			dminOut:  4.9598525010503808e-024,
			dmin1Out: 9.2339683204333278e-002,
			dmin2Out: 0.20112464802017624,
			dnOut:    4.9598525010503808e-024,
			dnm1Out:  9.2339683204333278e-002,
			dnm2Out:  0.20112464802017624,
		},
		{
			z:        []float64{0.1749149979302717e+01, 0.1749149979302717e+01, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.1117154804361126e+01, 0.1073347337297776e+01, 0.4438011869593646e-01, 0.4380746706334982e-01, 0.1178029724550133e+01, 0.1131758262754287e+01, 0.4427372589519608e-01, 0.9065158049178278e-01, 0.5998724677255393e+00, 0.5753431417100738e+00, 0.1369551627869799e+00, 0.6880305191066147e-01, 0.1150450694942405e+01, 0.1194069582486762e+01, 0.1142789997988411e+00, 0.9333627524262254e-01, 0.1303260395526232e+01, 0.1408587972834190e+01, 0.1155784656615063e-01, 0.8951422490882578e-02, 0.1671278822811555e+01, 0.1682736313985311e+01, 0.2916285016424428e-04, 0.1003553923945662e-03, 0.4993103973382956e+00, 0.4856665170586981e+00, 0.1200921574220688e-01, 0.1367304312976171e-01, 0.4396883475635844e+00, 0.4385509668224923e+00, 0.1138121911768345e-01, 0.1314659648329904e-01, 0.4050999534208807e+00, 0.3806452440729016e+00, 0.1951852336551518e-01, 0.3583592846566260e-01, 0.2011877025290218e+00, 0.2206431713856914e+00, 0.2894937755625969e-04, 0.6305450884558253e-04, 0.9233968320433328e-01, 0.9236863258188953e-01, 0.3202909346606844e-42, 0.5962992535266723e-20},
			i0:       1,
			n0:       12,
			pp:       0,
			tau:      9.2159326345418235e-002,
			sigma:    0.73182753419913071,
			i0Out:    1,
			n0Out:    12,
			ppOut:    0,
			tauOut:   9.2159326345418235e-002,
			sigmaOut: 0.73182753419913071,
			dminOut:  1.5272797578758102e-004,
			dmin1Out: 9.6724015270930774e-002,
			dmin2Out: 0.29962742013672317,
			dnOut:    1.5272797578758102e-004,
			dnm1Out:  9.6724015270930774e-002,
			dnm2Out:  0.29962742013672317,
		},
		{
			z:        []float64{0.1749149979302717e+01, 0.1656990652957299e+01, 0.8676862906242004e-16, 0.5850002270797901e-16, 0.1117154804361126e+01, 0.1069375596711644e+01, 0.4438011869593646e-01, 0.4888936980013561e-01, 0.1178029724550133e+01, 0.1081254754299776e+01, 0.4427372589519608e-01, 0.2456274906772994e-01, 0.5998724677255393e+00, 0.6201055550993710e+00, 0.1369551627869799e+00, 0.2540860356894923e+00, 0.1150450694942405e+01, 0.9184843327063353e+00, 0.1142789997988411e+00, 0.1621533314992303e+00, 0.1303260395526232e+01, 0.1060505584247734e+01, 0.1155784656615063e-01, 0.1821431635083262e-01, 0.1671278822811555e+01, 0.1560934342965469e+01, 0.2916285016424428e-04, 0.9328588590960435e-05, 0.4993103973382956e+00, 0.4191509581464933e+00, 0.1200921574220688e-01, 0.1259763844648080e-01, 0.4396883475635844e+00, 0.3463126018893689e+00, 0.1138121911768345e-01, 0.1331320693873929e-01, 0.4050999534208807e+00, 0.3191459435022383e+00, 0.1951852336551518e-01, 0.1230436091267282e-01, 0.2011877025290218e+00, 0.9675296464848704e-01, 0.2894937755625969e-04, 0.2762888312745317e-04, 0.9233968320433328e-01, 0.1527279757875810e-03, 0.3202909346606844e-42, 0.5850002270797901e-16, 0.7318275341991307e+00},
			i0:       1,
			n0:       12,
			pp:       1,
			tau:      1.5267965277267402e-004,
			sigma:    0.82398686054454895,
			i0Out:    1,
			n0Out:    12,
			ppOut:    1,
			tauOut:   1.5267965277267402e-004,
			sigmaOut: 0.82398686054454895,
			dminOut:  2.8998364833248812e-009,
			dmin1Out: 9.2869952038417761e-002,
			dmin2Out: 0.30683160095299705,
			dnOut:    2.8998364833248812e-009,
			dnm1Out:  9.2869952038417761e-002,
			dnm2Out:  0.30683160095299705,
		},
		{
			z:        []float64{0.1656837973304527e+01, 0.1656990652957299e+01, 0.3775776370348287e-16, 0.5850002270797901e-16, 0.1118112286859007e+01, 0.1069375596711644e+01, 0.4727777715386321e-01, 0.4888936980013561e-01, 0.1058387046560870e+01, 0.1081254754299776e+01, 0.1439123541326830e-01, 0.2456274906772994e-01, 0.8596476757228223e+00, 0.6201055550993710e+00, 0.2714763844897646e+00, 0.2540860356894923e+00, 0.8090086000630283e+00, 0.9184843327063353e+00, 0.2125620340079330e+00, 0.1621533314992303e+00, 0.8660051869378607e+00, 0.1060505584247734e+01, 0.3283046378299830e-01, 0.1821431635083262e-01, 0.1527960528118289e+01, 0.1560934342965469e+01, 0.2559023465659061e-05, 0.9328588590960435e-05, 0.4315933579167358e+00, 0.4191509581464933e+00, 0.1010840613748274e-01, 0.1259763844648080e-01, 0.3493647230378528e+00, 0.3463126018893689e+00, 0.1216166289646864e-01, 0.1331320693873929e-01, 0.3191359618656699e+00, 0.3191459435022383e+00, 0.3730332957296599e-02, 0.1230436091267282e-01, 0.9289758092154521e-01, 0.9675296464848704e-01, 0.4542317842367971e-07, 0.2762888312745317e-04, 0.2899836483324881e-08, 0.1527279757875810e-03, 0.3775776370348287e-16, 0.5850002270797901e-16},
			i0:       1,
			n0:       12,
			pp:       0,
			tau:      2.8998350258011044e-009,
			sigma:    0.82413954019732161,
			i0Out:    1,
			n0Out:    12,
			ppOut:    0,
			tauOut:   2.8998350258011044e-009,
			sigmaOut: 0.82413954019732161,
			dminOut:  2.2456987483859239e-017,
			dmin1Out: 9.1786476970678488e-002,
			dmin2Out: 0.30815750788579249,
			dnOut:    2.2456987483859239e-017,
			dnm1Out:  9.1786476970678488e-002,
			dnm2Out:  0.30815750788579249,
		},
		{
			z:        []float64{0.1656837973304527e+01, 0.1656837970404692e+01, 0.3775776370348287e-16, 0.2548071704976161e-16, 0.1118112286859007e+01, 0.1165390061113035e+01, 0.4727777715386321e-01, 0.4293685745187325e-01, 0.1058387046560870e+01, 0.1029841421622430e+01, 0.1439123541326830e-01, 0.1201290976848257e-01, 0.8596476757228223e+00, 0.1119111147544269e+01, 0.2714763844897646e+00, 0.1962510428460807e+00, 0.8090086000630283e+00, 0.8253195883250457e+00, 0.2125620340079330e+00, 0.2230406579474441e+00, 0.8660051869378607e+00, 0.6757949898735797e+00, 0.3283046378299830e-01, 0.7422909836846008e-01, 0.1527960528118289e+01, 0.1453733985873459e+01, 0.2559023465659061e-05, 0.7597383986781573e-06, 0.4315933579167358e+00, 0.4417010014159849e+00, 0.1010840613748274e-01, 0.7995273950601426e-02, 0.3493647230378528e+00, 0.3535311090838850e+00, 0.1216166289646864e-01, 0.1097845108004235e-01, 0.3191359618656699e+00, 0.3118878408430891e+00, 0.3730332957296599e-02, 0.1111101051031713e-02, 0.9289758092154521e-01, 0.9178652239385691e-01, 0.4542317842367971e-07, 0.1435066789177947e-14, 0.2899836483324881e-08, 0.2245698748385924e-16, 0.3775776370348287e-16, 0.2548071704976161e-16, 0.7318275341991307e+00},
			i0:       1,
			n0:       12,
			pp:       1,
			tau:      2.2456987483858885e-017,
			sigma:    0.82413954309715665,
			i0Out:    1,
			n0Out:    12,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.82413954309715665,
			dminOut:  2.2456987483858888e-017,
			dmin1Out: 9.1450429866798411e-002,
			dmin2Out: 0.30232944966555197,
			dnOut:    2.2456987483858888e-017,
			dnm1Out:  9.1450429866798411e-002,
			dnm2Out:  0.30232944966555197,
		},
		{
			z:        []float64{0.1656837970404692e+01, 0.1656837970404692e+01, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.1208326918564908e+01, 0.1165390061113035e+01, 0.3659452888027460e-01, 0.4293685745187325e-01, 0.1005259802510638e+01, 0.1029841421622430e+01, 0.1337343958524596e-01, 0.1201290976848257e-01, 0.1301988750805104e+01, 0.1119111147544269e+01, 0.1244018658302015e+00, 0.1962510428460807e+00, 0.9239583804422884e+00, 0.8253195883250457e+00, 0.1631347930486186e+00, 0.2230406579474441e+00, 0.5868892951934211e+00, 0.6757949898735797e+00, 0.1838666404767374e+00, 0.7422909836846008e-01, 0.1269868105135121e+01, 0.1453733985873459e+01, 0.2642614694812039e-06, 0.7597383986781573e-06, 0.4496960111051168e+00, 0.4417010014159849e+00, 0.6285530663790794e-02, 0.7995273950601426e-02, 0.3582240295001366e+00, 0.3535311090838850e+00, 0.9558391177537082e-02, 0.1097845108004235e-01, 0.3034405507165837e+00, 0.3118878408430891e+00, 0.3360925270585024e-03, 0.1111101051031713e-02, 0.9145042986679984e-01, 0.9178652239385691e-01, 0.3524015903480299e-30, 0.1435066789177947e-14},
			i0:       1,
			n0:       11,
			pp:       0,
			tau:      9.1173077708044642e-002,
			sigma:    0.82413954309715665,
			i0Out:    1,
			n0Out:    11,
			ppOut:    0,
			tauOut:   9.1173077708044642e-002,
			sigmaOut: 0.82413954309715665,
			dminOut:  1.2510327751137640e-004,
			dmin1Out: 0.20154260650453434,
			dmin2Out: 0.26087886618071221,
			dnOut:    1.2510327751137640e-004,
			dnm1Out:  0.20154260650453434,
			dnm2Out:  0.26087886618071221,
		},
		{
			z:        []float64{0.1656837970404692e+01, 0.1565664892696647e+01, 0.1792267857826344e-16, 0.1383211380667930e-16, 0.1208326918564908e+01, 0.1153748369737138e+01, 0.3659452888027460e-01, 0.3188477647299814e-01, 0.1005259802510638e+01, 0.8955753879148407e+00, 0.1337343958524596e-01, 0.1944232516271161e-01, 0.1301988750805104e+01, 0.1315775213764550e+01, 0.1244018658302015e+00, 0.8735697805677015e-01, 0.9239583804422884e+00, 0.9085631177260922e+00, 0.1631347930486186e+00, 0.1053774491236745e+00, 0.5868892951934211e+00, 0.5742054088384394e+00, 0.1838666404767374e+00, 0.4066251880351924e+00, 0.1269868105135121e+01, 0.7720701036533529e+00, 0.2642614694812039e-06, 0.1539203864417861e-06, 0.4496960111051168e+00, 0.3648083101404764e+00, 0.6285530663790794e-02, 0.6172085611379779e-02, 0.3582240295001366e+00, 0.2704372573582493e+00, 0.9558391177537082e-02, 0.1072486650400470e-01, 0.3034405507165837e+00, 0.2018786990315928e+00, 0.3360925270585024e-03, 0.1522488812438262e-03, 0.9145042986679984e-01, 0.1251032775113764e-03, 0.3524015903480299e-30, 0.1383211380667930e-16, 0.8241395430971566e+00},
			i0:       1,
			n0:       11,
			pp:       1,
			tau:      1.2498058228587147e-004,
			sigma:    0.91531262080520126,
			i0Out:    1,
			n0Out:    11,
			ppOut:    1,
			tauOut:   1.2498058228587147e-004,
			sigmaOut: 0.91531262080520126,
			dminOut:  2.4554386425680716e-008,
			dmin1Out: 0.19392428663386141,
			dmin2Out: 0.26581143244488259,
			dnOut:    2.4554386425680716e-008,
			dnm1Out:  0.19392428663386141,
			dnm2Out:  0.26581143244488259,
		},
		{
			z:        []float64{0.1565539912114361e+01, 0.1565664892696647e+01, 0.1019378594629470e-16, 0.1383211380667930e-16, 0.1185508165627851e+01, 0.1153748369737138e+01, 0.2408690373149840e-01, 0.3188477647299814e-01, 0.8908058287637680e+00, 0.8955753879148407e+00, 0.2871751477260568e-01, 0.1944232516271161e-01, 0.1374289696466428e+01, 0.1315775213764550e+01, 0.5775298217141787e-01, 0.8735697805677015e-01, 0.9560626040960629e+00, 0.9085631177260922e+00, 0.6328905763825028e-01, 0.1053774491236745e+00, 0.9174165586530958e+00, 0.5742054088384394e+00, 0.3422034931823232e+00, 0.4066251880351924e+00, 0.4297417838091302e+00, 0.7720701036533529e+00, 0.1306631986684747e-06, 0.1539203864417861e-06, 0.3708552845063717e+00, 0.3648083101404764e+00, 0.4500844331080801e-02, 0.6172085611379779e-02, 0.2765362989488873e+00, 0.2704372573582493e+00, 0.7829431815445537e-02, 0.1072486650400470e-01, 0.1940765355151052e+00, 0.2018786990315928e+00, 0.9814083907923291e-07, 0.1522488812438262e-03, 0.2455438642568072e-07, 0.1251032775113764e-03, 0.1019378594629470e-16, 0.1383211380667930e-16},
			i0:       1,
			n0:       11,
			pp:       0,
			tau:      2.4554370888251911e-008,
			sigma:    0.91543760138748709,
			i0Out:    1,
			n0Out:    11,
			ppOut:    0,
			tauOut:   2.4554370888251911e-008,
			sigmaOut: 0.91543760138748709,
			dminOut:  2.7649302792337775e-015,
			dmin1Out: 0.18866996369915248,
			dmin2Out: 0.27322036360404983,
			dnOut:    2.7649302792337775e-015,
			dnm1Out:  0.18866996369915248,
			dnm2Out:  0.27322036360404983,
		},
		{
			z:        []float64{0.1565539912114361e+01, 0.1565539887559990e+01, 0.1019378594629470e-16, 0.7719264500395369e-17, 0.1185508165627851e+01, 0.1209595044804978e+01, 0.2408690373149840e-01, 0.1773879145177052e-01, 0.8908058287637680e+00, 0.9017845275302323e+00, 0.2871751477260568e-01, 0.4376453959373496e-01, 0.1374289696466428e+01, 0.1388278114489740e+01, 0.5775298217141787e-01, 0.3977262621431848e-01, 0.9560626040960629e+00, 0.9795790109656241e+00, 0.6328905763825028e-01, 0.5927283946360358e-01, 0.9174165586530958e+00, 0.1200347187817445e+01, 0.3422034931823232e+00, 0.1225138368952072e+00, 0.4297417838091302e+00, 0.3072280530227507e+00, 0.1306631986684747e-06, 0.1577236754259593e-06, 0.3708552845063717e+00, 0.3753559465594062e+00, 0.4500844331080801e-02, 0.3315910790466669e-02, 0.2765362989488873e+00, 0.2810497954194954e+00, 0.7829431815445537e-02, 0.5406547261581845e-02, 0.1940765355151052e+00, 0.1886700618399915e+00, 0.9814083907923291e-07, 0.1277249852674422e-13, 0.2455438642568072e-07, 0.2764930279233778e-14, 0.1019378594629470e-16, 0.7719264500395369e-17, 0.8241395430971566e+00},
			i0:       1,
			n0:       11,
			pp:       1,
			tau:      2.7649302792335523e-015,
			sigma:    0.91543762594185796,
			i0Out:    1,
			n0Out:    11,
			ppOut:    1,
			tauOut:   2.7649302792335523e-015,
			sigmaOut: 0.91543762594185796,
			dminOut:  3.4709879829724519e-029,
			dmin1Out: 0.18507826404867261,
			dmin2Out: 0.27717487004787600,
			dnOut:    3.4709879829724519e-029,
			dnm1Out:  0.18507826404867261,
			dnm2Out:  0.27858872904535659,
		},
		{
			z:        []float64{0.1565539887559988e+01, 0.1565539887559990e+01, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.1227333836256746e+01, 0.1209595044804978e+01, 0.1303359134714337e-01, 0.1773879145177052e-01, 0.9325154757768211e+00, 0.9017845275302323e+00, 0.6515425651042277e-01, 0.4376453959373496e-01, 0.1362896484193633e+01, 0.1388278114489740e+01, 0.2858649229958118e-01, 0.3977262621431848e-01, 0.1010265358129644e+01, 0.9795790109656241e+00, 0.7042504782685149e-01, 0.5927283946360358e-01, 0.1252435976885798e+01, 0.1200347187817445e+01, 0.3005318297487194e-01, 0.1225138368952072e+00, 0.2771750277715515e+00, 0.3072280530227507e+00, 0.2135925446109612e-06, 0.1577236754259593e-06, 0.3786716437573254e+00, 0.3753559465594062e+00, 0.2461066374135985e-02, 0.3315910790466669e-02, 0.2839952763069384e+00, 0.2810497954194954e+00, 0.3591797791316158e-02, 0.5406547261581845e-02, 0.1850782640486854e+00, 0.1886700618399915e+00, 0.1908115364037247e-27, 0.1277249852674422e-13},
			i0:       1,
			n0:       10,
			pp:       0,
			tau:      9.2539132024336307e-002,
			sigma:    0.91543762594186073,
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			tauOut:   9.2539132024336307e-002,
			sigmaOut: 0.91543762594186073,
			dminOut:  8.9088074504151571e-002,
			dmin1Out: 0.17705256406949207,
			dmin2Out: 0.17705256406949207,
			dnOut:    8.9088074504151571e-002,
			dnm1Out:  0.18903428744201686,
			dnm2Out:  0.28613205491191551,
		},
		{
			z:        []float64{0.1565539887559988e+01, 0.1473000755535651e+01, 0.5964194309842799e-17, 0.4969486576955697e-17, 0.1227333836256746e+01, 0.1147828295579553e+01, 0.1303359134714337e-01, 0.1058871408116432e-01, 0.9325154757768211e+00, 0.8945418861817434e+00, 0.6515425651042277e-01, 0.9926701980086396e-01, 0.1362896484193633e+01, 0.1199676824668014e+01, 0.2858649229958118e-01, 0.2407310226126826e-01, 0.1010265358129644e+01, 0.9640781716708908e+00, 0.7042504782685149e-01, 0.9148932748822959e-01, 0.1252435976885798e+01, 0.1098460700348104e+01, 0.3005318297487194e-01, 0.7583331677723075e-02, 0.2771750277715515e+00, 0.1770527776620367e+00, 0.2135925446109612e-06, 0.4568210735249311e-06, 0.3786716437573254e+00, 0.2885931212860515e+00, 0.2461066374135985e-02, 0.2421856840585221e-02, 0.2839952763069384e+00, 0.1926260852333330e+00, 0.3591797791316158e-02, 0.3451057520197492e-02, 0.1850782640486854e+00, 0.8908807450415157e-01, 0.1908115364037247e-27, 0.4969486576955697e-17, 0.9154376259418607e+00},
			i0:       1,
			n0:       10,
			pp:       1,
			tau:      7.5387921832037685e-002,
			sigma:    1.0079767579661971,
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			tauOut:   7.5387921832037685e-002,
			sigmaOut: 1.0079767579661971,
			dminOut:  1.1106216559126303e-002,
			dmin1Out: 0.10021441433509834,
			dmin2Out: 0.10021441433509834,
			dnOut:    1.1106216559126303e-002,
			dnm1Out:  0.11507463320059727,
			dnm2Out:  0.21320388392650824,
		},
		{
			z:        []float64{0.1397612833703614e+01, 0.1473000755535651e+01, 0.4081328655531061e-17, 0.4969486576955697e-17, 0.1083029087828680e+01, 0.1147828295579553e+01, 0.8745885380967966e-02, 0.1058871408116432e-01, 0.9096750987696016e+00, 0.8945418861817434e+00, 0.1309130515609722e+00, 0.9926701980086396e-01, 0.1017448953536272e+01, 0.1199676824668014e+01, 0.2281033592282574e-01, 0.2407310226126826e-01, 0.9573692414042568e+00, 0.9640781716708908e+00, 0.1049724875218357e+00, 0.9148932748822959e-01, 0.9256836226719531e+00, 0.1098460700348104e+01, 0.1450441494900679e-02, 0.7583331677723075e-02, 0.1002148711561719e+00, 0.1770527776620367e+00, 0.1315527505616969e-05, 0.4568210735249311e-06, 0.2156257407670935e+00, 0.2885931212860515e+00, 0.2163530200698068e-02, 0.2421856840585221e-02, 0.1185256907207948e+00, 0.1926260852333330e+00, 0.2593936112987583e-02, 0.3451057520197492e-02, 0.1110621655912630e-01, 0.8908807450415157e-01, 0.4081328655531061e-17, 0.4969486576955697e-17},
			i0:       1,
			n0:       10,
			pp:       0,
			tau:      9.2006637361187298e-003,
			sigma:    1.0833646797982348,
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			tauOut:   9.2006637361187298e-003,
			sigmaOut: 1.0833646797982348,
			dminOut:  1.6452860548183366e-003,
			dmin1Out: 9.0837695870445614e-002,
			dmin2Out: 9.0837695870445614e-002,
			dnOut:    1.6452860548183366e-003,
			dnm1Out:  0.10809563217808144,
			dnm2Out:  0.20642195434625446,
		},
		{
			z:        []float64{0.1397612833703614e+01, 0.1388412169967495e+01, 0.4081328655531061e-17, 0.3183635051997810e-17, 0.1083029087828680e+01, 0.1082574309473529e+01, 0.8745885380967966e-02, 0.7349069784991225e-02, 0.9096750987696016e+00, 0.1024038416809464e+01, 0.1309130515609722e+00, 0.1300706547025319e+00, 0.1017448953536272e+01, 0.9009879710204475e+00, 0.2281033592282574e-01, 0.2423774201322422e-01, 0.9573692414042568e+00, 0.1028903323176749e+01, 0.1049724875218357e+00, 0.9444163542020824e-01, 0.9256836226719531e+00, 0.8234917650105269e+00, 0.1450441494900679e-02, 0.1765115496075164e-03, 0.1002148711561719e+00, 0.9083901139795122e-01, 0.1315527505616969e-05, 0.3122684720284652e-05, 0.2156257407670935e+00, 0.2085854845469525e+00, 0.2163530200698068e-02, 0.1229394806594584e-02, 0.1185256907207948e+00, 0.1106895682910690e+00, 0.2593936112987583e-02, 0.2602667681892373e-03, 0.1110621655912630e-01, 0.1645286054818337e-02, 0.4081328655531061e-17, 0.3183635051997810e-17, 0.9154376259418607e+00},
			i0:       1,
			n0:       10,
			pp:       1,
			tau:      1.5594219393746818e-003,
			sigma:    1.0925653435343534,
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			tauOut:   1.5594219393746818e-003,
			sigmaOut: 1.0925653435343534,
			dminOut:  8.1926049685687600e-005,
			dmin1Out: 8.9258234858565516e-002,
			dmin2Out: 8.9258234858565516e-002,
			dnOut:    8.1926049685687600e-005,
			dnm1Out:  0.10847668953277810,
			dnm2Out:  0.20701876553386761,
		},
		{
			z:        []float64{0.1386852748028120e+01, 0.1388412169967495e+01, 0.2485138759635906e-17, 0.3183635051997810e-17, 0.1088363957319145e+01, 0.1082574309473529e+01, 0.6914717946174950e-02, 0.7349069784991225e-02, 0.1145634931626446e+01, 0.1024038416809464e+01, 0.1022944500333619e+00, 0.1300706547025319e+00, 0.8213718410609353e+00, 0.9009879710204475e+00, 0.3036175828902931e-01, 0.2423774201322422e-01, 0.1091423778368554e+01, 0.1028903323176749e+01, 0.7125729765473905e-01, 0.9444163542020824e-01, 0.7508515569660207e+00, 0.8234917650105269e+00, 0.2135460001102832e-04, 0.1765115496075164e-03, 0.8926135754328580e-01, 0.9083901139795122e-01, 0.7297073710223142e-05, 0.3122684720284652e-05, 0.2082481603404622e+00, 0.2085854845469525e+00, 0.6534568189162364e-03, 0.1229394806594584e-02, 0.1087369563009673e+00, 0.1106895682910690e+00, 0.3938065757966957e-05, 0.2602667681892373e-03, 0.8192604968568760e-04, 0.1645286054818337e-02, 0.2485138759635906e-17, 0.3183635051997810e-17},
			i0:       1,
			n0:       10,
			pp:       0,
			tau:      8.1416944590412474e-005,
			sigma:    1.0941247654737283,
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			tauOut:   8.1416944590412474e-005,
			sigmaOut: 1.0941247654737283,
			dminOut:  5.0612658152462498e-007,
			dmin1Out: 8.9177229293753768e-002,
			dmin2Out: 8.9177229293753768e-002,
			dnOut:    5.0612658152462498e-007,
			dnm1Out:  0.10831524323681536,
			dnm2Out:  0.20814970454128662,
		},
		{
			z:        []float64{0.1386852748028120e+01, 0.1386771331083530e+01, 0.2485138759635906e-17, 0.1950383162890474e-17, 0.1088363957319145e+01, 0.1095197258320730e+01, 0.6914717946174950e-02, 0.7233164949324962e-02, 0.1145634931626446e+01, 0.1240614799765893e+01, 0.1022944500333619e+00, 0.6772592167212049e-01, 0.8213718410609353e+00, 0.7839262607332537e+00, 0.3036175828902931e-01, 0.4227125255215914e-01, 0.1091423778368554e+01, 0.1120328406526543e+01, 0.7125729765473905e-01, 0.4775711530437247e-01, 0.7508515569660207e+00, 0.7030343793170689e+00, 0.2135460001102832e-04, 0.2711304941630149e-05, 0.8926135754328580e-01, 0.8918452636746399e-01, 0.7297073710223142e-05, 0.1703885458517271e-04, 0.2082481603404622e+00, 0.2088031613602029e+00, 0.6534568189162364e-03, 0.3402961195615630e-03, 0.1087369563009673e+00, 0.1083191813025733e+00, 0.3938065757966957e-05, 0.2978513750500819e-08, 0.8192604968568760e-04, 0.5061265815246250e-06, 0.2485138759635906e-17, 0.1950383162890474e-17, 0.9154376259418607e+00},
			i0:       1,
			n0:       10,
			pp:       1,
			tau:      5.0604049633765406e-007,
			sigma:    1.0942061824183187,
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			tauOut:   5.0604049633765406e-007,
			sigmaOut: 1.0942061824183187,
			dminOut:  8.6071246971392626e-011,
			dmin1Out: 8.9183660885577137e-002,
			dmin2Out: 8.9183660885577137e-002,
			dnOut:    8.6071246971392626e-011,
			dnm1Out:  0.10814239569959758,
			dnm2Out:  0.20876277035564564,
		},
		{
			z:        []float64{0.1386770825043033e+01, 0.1386771331083530e+01, 0.1540308069724697e-17, 0.1950383162890474e-17, 0.1102429917229558e+01, 0.1095197258320730e+01, 0.8139811288713328e-02, 0.7233164949324962e-02, 0.1300200404108803e+01, 0.1240614799765893e+01, 0.4083380405309871e-01, 0.6772592167212049e-01, 0.7853632031918177e+00, 0.7839262607332537e+00, 0.6030036143936175e-01, 0.4227125255215914e-01, 0.1107784654351057e+01, 0.1120328406526543e+01, 0.3030814137396716e-01, 0.4775711530437247e-01, 0.6727284432075471e+00, 0.7030343793170689e+00, 0.3594413905024206e-06, 0.2711304941630149e-05, 0.8920069974016231e-01, 0.8918452636746399e-01, 0.3988496406087022e-04, 0.1703885458517271e-04, 0.2091030664752072e+00, 0.2088031613602029e+00, 0.1762795624794033e-03, 0.3402961195615630e-03, 0.1081423986781113e+00, 0.1083191813025733e+00, 0.1393999949133917e-13, 0.2978513750500819e-08, 0.8607124697139263e-10, 0.5061265815246250e-06, 0.1540308069724697e-17, 0.1950383162890474e-17},
			i0:       1,
			n0:       10,
			pp:       0,
			tau:      8.6071215292546838e-011,
			sigma:    1.0942066884588149,
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			tauOut:   8.6071215292546838e-011,
			sigmaOut: 1.0942066884588149,
			dminOut:  3.1678834699165494e-017,
			dmin1Out: 8.9200650586607991e-002,
			dmin2Out: 8.9200650586607991e-002,
			dnOut:    3.1678834699165494e-017,
			dnm1Out:  0.10805126770967630,
			dnm2Out:  0.20900961033533805,
		},
		{
			z:        []float64{0.1386770825043033e+01, 0.1386770824956962e+01, 0.1540308069724697e-17, 0.1224486171222500e-17, 0.1102429917229558e+01, 0.1110569728432201e+01, 0.8139811288713328e-02, 0.9529690622753703e-02, 0.1300200404108803e+01, 0.1331504517453077e+01, 0.4083380405309871e-01, 0.2408506071837550e-01, 0.7853632031918177e+00, 0.8215785038267327e+00, 0.6030036143936175e-01, 0.8130667336500198e-01, 0.1107784654351057e+01, 0.1056786122273951e+01, 0.3030814137396716e-01, 0.1929354325655847e-01, 0.6727284432075471e+00, 0.6534352593063080e+00, 0.3594413905024206e-06, 0.4906748310831983e-07, 0.8920069974016231e-01, 0.8924053555066887e-01, 0.3988496406087022e-04, 0.9345605379795243e-04, 0.2091030664752072e+00, 0.2091858898978174e+00, 0.1762795624794033e-03, 0.9113088236382798e-04, 0.1081423986781113e+00, 0.1080512677096902e+00, 0.1393999949133917e-13, 0.1110429488179469e-22, 0.8607124697139263e-10, 0.3167883469916549e-16, 0.1540308069724697e-17, 0.1224486171222500e-17, 0.9154376259418607e+00},
			i0:       1,
			n0:       10,
			pp:       1,
			tau:      3.1678834698836348e-017,
			sigma:    1.0942066885448862,
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.0942066885448862,
			dminOut:  3.1678834699165494e-017,
			dmin1Out: 8.9240528714785800e-002,
			dmin2Out: 8.9240528714785800e-002,
			dnOut:    3.1678834699165494e-017,
			dnm1Out:  0.10800416691102718,
			dnm2Out:  0.20896705170504334,
		},
		{
			z:        []float64{0.1386770824956962e+01, 0.1386770824956962e+01, 0.9806070694382835e-18, 0.1224486171222500e-17, 0.1120099419054954e+01, 0.1110569728432201e+01, 0.1132830345080667e-01, 0.9529690622753703e-02, 0.1344261274720646e+01, 0.1331504517453077e+01, 0.1472018016266302e-01, 0.2408506071837550e-01, 0.8881649970290716e+00, 0.8215785038267327e+00, 0.9674301998819117e-01, 0.8130667336500198e-01, 0.9793366455423188e+00, 0.1056786122273951e+01, 0.1287308250760436e-01, 0.1929354325655847e-01, 0.6405622258661866e+00, 0.6534352593063080e+00, 0.6835883063177366e-08, 0.4906748310831983e-07, 0.8933398476858376e-01, 0.8924053555066887e-01, 0.2188381927740679e-03, 0.9345605379795243e-04, 0.2090581825874072e+00, 0.2091858898978174e+00, 0.4710079866305792e-04, 0.9113088236382798e-04, 0.1080041669110272e+00, 0.1080512677096902e+00, 0.3257014354834561e-38, 0.1110429488179469e-22},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      2.2310132178696450e-002,
			sigma:    1.0942066885448862,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   2.2310132178696450e-002,
			sigmaOut: 1.0942066885448862,
			dminOut:  6.7023851586769906e-002,
			dmin1Out: 6.7023851586769906e-002,
			dmin2Out: 6.7023851586769906e-002,
			dnOut:    8.5666701693601133e-002,
			dnm1Out:  0.18606768031923254,
			dnm2Out:  6.7023851586769906e-002,
		},
		{
			z:        []float64{0.1386770824956962e+01, 0.1364460692778266e+01, 0.9806070694382835e-18, 0.8049901434408684e-18, 0.1120099419054954e+01, 0.1109117590327065e+01, 0.1132830345080667e-01, 0.1373001363427395e-01, 0.1344261274720646e+01, 0.1322941309070339e+01, 0.1472018016266302e-01, 0.9882485852396858e-02, 0.8881649970290716e+00, 0.9527153989861693e+00, 0.9674301998819117e-01, 0.9944626147083412e-01, 0.9793366455423188e+00, 0.8704533344003925e+00, 0.1287308250760436e-01, 0.9473236598617136e-02, 0.6405622258661866e+00, 0.6087788639247561e+00, 0.6835883063177366e-08, 0.1003117403762534e-08, 0.8933398476858376e-01, 0.6724268977954398e-01, 0.2188381927740679e-03, 0.6803700894781421e-03, 0.2090581825874072e+00, 0.1861147811178956e+00, 0.4710079866305792e-04, 0.2733303872960605e-04, 0.1080041669110272e+00, 0.8566670169360113e-01, 0.3257014354834561e-38, 0.8049901434408684e-18, 0.1094206688544886e+01},
			i0:       1,
			n0:       9,
			pp:       1,
			tau:      6.4730147312741043e-002,
			sigma:    1.1165168207235825,
			i0Out:    1,
			n0Out:    9,
			ppOut:    1,
			tauOut:   6.4730147312741043e-002,
			sigmaOut: 1.1165168207235825,
			dminOut:  2.5125423409859404e-003,
			dmin1Out: 2.5125423409859404e-003,
			dmin2Out: 2.5125423409859404e-003,
			dnOut:    2.0907912921905053e-002,
			dnm1Out:  8.1725879285544201e-002,
			dnm2Out:  2.5125423409859404e-003,
		},
		{
			z:        []float64{0.1299730545465525e+01, 0.1364460692778266e+01, 0.6869337119490330e-18, 0.8049901434408684e-18, 0.1058117456648598e+01, 0.1109117590327065e+01, 0.1716633828952343e-01, 0.1373001363427395e-01, 0.1250927309320471e+01, 0.1322941309070339e+01, 0.7526573591998700e-02, 0.9882485852396858e-02, 0.9799049395522637e+00, 0.9527153989861693e+00, 0.8833849733474472e-01, 0.9944626147083412e-01, 0.7268579263515238e+00, 0.8704533344003925e+00, 0.7934296380510910e-02, 0.9473236598617136e-02, 0.5361144212346215e+00, 0.6087788639247561e+00, 0.1258170079408221e-09, 0.1003117403762534e-08, 0.3192912430464083e-02, 0.6724268977954398e-01, 0.3965875451961033e-01, 0.6803700894781421e-03, 0.8175321232427381e-01, 0.1861147811178956e+00, 0.2864145895504550e-04, 0.2733303872960605e-04, 0.2090791292190505e-01, 0.8566670169360113e-01, 0.6869337119490330e-18, 0.8049901434408684e-18},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      2.3478378904869292e-003,
			sigma:    1.1812469680363236,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   2.3478378904869292e-003,
			sigmaOut: 1.1812469680363236,
			dminOut:  -6.4213352031532861e-004,
			dmin1Out: -6.4213352031532861e-004,
			dmin2Out: 8.4507453921550072e-004,
			dnOut:    1.9536180783236046e-002,
			dnm1Out:  -6.4213352031532861e-004,
			dnm2Out:  8.4507453921550072e-004,
		},
		{
			z:        []float64{0.1299730545465525e+01, 0.1297382707575038e+01, 0.6869337119490330e-18, 0.5602483738451179e-18, 0.1058117456648598e+01, 0.1072935957047634e+01, 0.1716633828952343e-01, 0.2001409424891253e-01, 0.1250927309320471e+01, 0.1236091950773070e+01, 0.7526573591998700e-02, 0.5966648869520194e-02, 0.9799049395522637e+00, 0.1059928950127001e+01, 0.8833849733474472e-01, 0.6057909540261972e-01, 0.7268579263515238e+00, 0.6718652894389281e+00, 0.7934296380510910e-02, 0.6331166051893822e-02, 0.5361144212346215e+00, 0.5274354174180577e+00, 0.1258170079408221e-09, 0.7616528495272331e-12, 0.3192912430464083e-02, 0.4050382905882583e-01, 0.3965875451961033e-01, 0.8004750795410222e-01, 0.8175321232427381e-01, -0.6134920613602832e-03, 0.2864145895504550e-04, -0.9761057518179271e-03, 0.2090791292190505e-01, 0.1953618078323605e-01, 0.6869337119490330e-18, 0.5602483738451179e-18},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      5.8695947262173229e-004,
			sigma:    1.1812469680363236,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   5.8695947262173229e-004,
			sigmaOut: 1.1812469680363236,
			dminOut:  2.6059529570832572e-003,
			dmin1Out: 2.6059529570832572e-003,
			dmin2Out: 2.6059529570832572e-003,
			dnOut:    2.0187357299561493e-002,
			dnm1Out:  4.4537716273149721e-003,
			dnm2Out:  2.6059529570832572e-003,
		},
		{
			z:        []float64{0.1299730545465525e+01, 0.1299143585992903e+01, 0.6869337119490330e-18, 0.5594890049187079e-18, 0.1058117456648598e+01, 0.1074696835465499e+01, 0.1716633828952343e-01, 0.1998130138542489e-01, 0.1250927309320471e+01, 0.1237885622054423e+01, 0.7526573591998700e-02, 0.5958003315736791e-02, 0.9799049395522637e+00, 0.1061698474098650e+01, 0.8833849733474472e-01, 0.6047812873071532e-01, 0.7268579263515238e+00, 0.6737271345286977e+00, 0.7934296380510910e-02, 0.6313669873066943e-02, 0.5361144212346215e+00, 0.5292137920147498e+00, 0.1258170079408221e-09, 0.7590933847144573e-12, 0.3192912430464083e-02, 0.4226470747669359e-01, 0.3965875451961033e-01, 0.7671248122433710e-01, 0.8175321232427381e-01, 0.4482413086270017e-02, 0.2864145895504550e-04, 0.1335961497218277e-03, 0.2090791292190505e-01, 0.2018735729956149e-01, 0.6869337119490330e-18, 0.5594890049187079e-18, 0.1094206688544886e+01},
			i0:       1,
			n0:       9,
			pp:       1,
			tau:      4.9324499328963489e-004,
			sigma:    1.1818339275089453,
			i0Out:    1,
			n0Out:    9,
			ppOut:    1,
			tauOut:   4.9324499328963489e-004,
			sigmaOut: 1.1818339275089453,
			dminOut:  1.0870277779300500e-003,
			dmin1Out: 1.0870277779300500e-003,
			dmin2Out: 4.1771462483342674e-002,
			dnOut:    1.7484624889890960e-002,
			dnm1Out:  1.0870277779300500e-003,
			dnm2Out:  4.1771462483342674e-002,
		},
		{
			z:        []float64{0.1298650340999613e+01, 0.1299143585992903e+01, 0.4630045856693428e-18, 0.5594890049187079e-18, 0.1094184891857634e+01, 0.1074696835465499e+01, 0.2260547177996662e-01, 0.1998130138542489e-01, 0.1220744908596903e+01, 0.1237885622054423e+01, 0.5181756634367578e-02, 0.5958003315736791e-02, 0.1116501601201708e+01, 0.1061698474098650e+01, 0.3649413160495897e-01, 0.6047812873071532e-01, 0.6430534278035160e+00, 0.6737271345286977e+00, 0.5195962000339362e-02, 0.6313669873066943e-02, 0.5235245850218799e+00, 0.5292137920147498e+00, 0.6128243213469890e-13, 0.7590933847144573e-12, 0.1184839437076798e+00, 0.4226470747669359e-01, 0.2902140315050332e-02, 0.7671248122433710e-01, 0.1220623927651878e-02, 0.4482413086270017e-02, 0.2209487416380896e-02, 0.1335961497218277e-03, 0.1748462488989096e-01, 0.2018735729956149e-01, 0.4630045856693428e-18, 0.5594890049187079e-18},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      8.6238530354903250e-004,
			sigma:    1.1823271725022351,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   8.6238530354903250e-004,
			sigmaOut: 1.1823271725022351,
			dminOut:  3.2884671240747138e-004,
			dmin1Out: 3.2884671240747138e-004,
			dmin2Out: 0.11762155840411674,
			dnOut:    1.4027859154437344e-003,
			dnm1Out:  3.2884671240747138e-004,
			dnm2Out:  0.11762155840411674,
		},
		{
			z:        []float64{0.1298650340999613e+01, 0.1297787955696064e+01, 0.4630045856693428e-18, 0.3903662538064461e-18, 0.1094184891857634e+01, 0.1115927978334052e+01, 0.2260547177996662e-01, 0.2472875948770642e-01, 0.1220744908596903e+01, 0.1200335520440015e+01, 0.5181756634367578e-02, 0.4819852017032840e-02, 0.1116501601201708e+01, 0.1147313495486085e+01, 0.3649413160495897e-01, 0.2045445862496274e-01, 0.6430534278035160e+00, 0.6269325458753436e+00, 0.5195962000339362e-02, 0.4338925882718484e-02, 0.5235245850218799e+00, 0.5183232738356738e+00, 0.6128243213469890e-13, 0.1400860159256393e-13, 0.1184839437076798e+00, 0.1205236987191671e+00, 0.2902140315050332e-02, 0.2939191169537379e-04, 0.1220623927651878e-02, 0.2538334128788367e-02, 0.2209487416380896e-02, 0.1521945367089819e-01, 0.1748462488989096e-01, 0.1402785915443734e-02, 0.4630045856693428e-18, 0.3903662538064461e-18, 0.1094206688544886e+01},
			i0:       1,
			n0:       9,
			pp:       1,
			tau:      1.8442717265434725e-004,
			sigma:    1.1831895578057841,
			i0Out:    1,
			n0Out:    9,
			ppOut:    1,
			tauOut:   1.8442717265434725e-004,
			sigmaOut: 1.1831895578057841,
			dminOut:  3.4295817345512611e-006,
			dmin1Out: 2.3532871395603098e-003,
			dmin2Out: 0.12033927154650945,
			dnOut:    3.4295817345512611e-006,
			dnm1Out:  2.3532871395603098e-003,
			dnm2Out:  0.12033927154650945,
		},
		{
			z:        []float64{0.1297603528523410e+01, 0.1297787955696064e+01, 0.3357116521683426e-18, 0.3903662538064461e-18, 0.1140472310649104e+01, 0.1115927978334052e+01, 0.2602676813137004e-01, 0.2472875948770642e-01, 0.1178944177153024e+01, 0.1200335520440015e+01, 0.4690536984322235e-02, 0.4819852017032840e-02, 0.1162892989954071e+01, 0.1147313495486085e+01, 0.1102729651913737e-01, 0.2045445862496274e-01, 0.6200597480662705e+00, 0.6269325458753436e+00, 0.3627015421456806e-02, 0.4338925882718484e-02, 0.5145118312415766e+00, 0.5183232738356738e+00, 0.3281495925457712e-14, 0.1400860159256393e-13, 0.1203686634582048e+00, 0.1205236987191671e+00, 0.6198165737098726e-06, 0.2939191169537379e-04, 0.1757274081045850e-01, 0.2538334128788367e-02, 0.1214929161054836e-02, 0.1521945367089819e-01, 0.3429581734551261e-05, 0.1402785915443734e-02, 0.3357116521683426e-18, 0.3903662538064461e-18},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      3.2077610710809750e-006,
			sigma:    1.1833739849784384,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   3.2077610710809750e-006,
			sigmaOut: 1.1833739849784384,
			dminOut:  3.3505760549073344e-012,
			dmin1Out: 1.7569442559804263e-002,
			dmin2Out: 0.12036545569713296,
			dnOut:    3.3505760549073344e-012,
			dnm1Out:  1.7569442559804263e-002,
			dnm2Out:  0.12036545569713296,
		},
		{
			z:        []float64{0.1297603528523410e+01, 0.1297600320762339e+01, 0.3357116521683426e-18, 0.2950599175525191e-18, 0.1140472310649104e+01, 0.1166495871019403e+01, 0.2602676813137004e-01, 0.2630451380147252e-01, 0.1178944177153024e+01, 0.1157326992574803e+01, 0.4690536984322235e-02, 0.4713095445958055e-02, 0.1162892989954071e+01, 0.1169203983266179e+01, 0.1102729651913737e-01, 0.5848066547299594e-02, 0.6200597480662705e+00, 0.6178354891793567e+00, 0.3627015421456806e-02, 0.3020451850239120e-02, 0.5145118312415766e+00, 0.5114881716302697e+00, 0.3281495925457712e-14, 0.7722354114894537e-15, 0.1203686634582048e+00, 0.1203660755137067e+00, 0.6198165737098726e-06, 0.9048958315991394e-07, 0.1757274081045850e-01, 0.1878437172085910e-01, 0.1214929161054836e-02, 0.2218173128942313e-06, 0.3429581734551261e-05, 0.3350576054907334e-11, 0.3357116521683426e-18, 0.2950599175525191e-18, 0.1094206688544886e+01},
			i0:       1,
			n0:       9,
			pp:       1,
			tau:      3.3505364896797715e-012,
			sigma:    1.1833771927395094,
			i0Out:    1,
			n0Out:    9,
			ppOut:    1,
			tauOut:   3.3505364896797715e-012,
			sigmaOut: 1.1833771927395094,
			dminOut:  2.0335799151090034e-023,
			dmin1Out: 1.8784357595683275e-002,
			dmin2Out: 0.12036607551035594,
			dnOut:    2.0335799151090034e-023,
			dnm1Out:  1.8784357595683275e-002,
			dnm2Out:  0.12036607551035594,
		},
		{
			z:        []float64{0.1297600320758988e+01, 0.1297600320762339e+01, 0.2652482201353177e-18, 0.2950599175525191e-18, 0.1192800384817525e+01, 0.1166495871019403e+01, 0.2552222839336001e-01, 0.2630451380147252e-01, 0.1136517859624050e+01, 0.1157326992574803e+01, 0.4848643531876122e-02, 0.4713095445958055e-02, 0.1170203406278252e+01, 0.1169203983266179e+01, 0.3087619670750762e-02, 0.5848066547299594e-02, 0.6177683213554945e+00, 0.6178354891793567e+00, 0.2500816796475141e-02, 0.3020451850239120e-02, 0.5089873548304448e+00, 0.5114881716302697e+00, 0.1826193617023393e-15, 0.7722354114894537e-15, 0.1203661659999391e+00, 0.1203660755137067e+00, 0.1412182528886294e-07, 0.9048958315991394e-07, 0.1878457941299617e-01, 0.1878437172085910e-01, 0.3956520722700361e-16, 0.2218173128942313e-06, 0.2033579915109003e-22, 0.3350576054907334e-11, 0.2652482201353177e-18, 0.2950599175525191e-18},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      2.0335799151089990e-023,
			sigma:    1.1833771927428600,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.1833771927428600,
			dminOut:  2.0335799151089993e-023,
			dmin1Out: 1.8784577209116738e-002,
			dmin2Out: 0.12036616599993906,
			dnOut:    2.0335799151089993e-023,
			dnm1Out:  1.8784577209116738e-002,
			dnm2Out:  0.12036616599993906,
		},
		{
			z:        []float64{0.1297600320758988e+01, 0.1297600320758988e+01, 0.2652482201353177e-18, 0.2438256017573345e-18, 0.1192800384817525e+01, 0.1218322613210885e+01, 0.2552222839336001e-01, 0.2380852827644004e-01, 0.1136517859624050e+01, 0.1117557974879486e+01, 0.4848643531876122e-02, 0.5077051306839188e-02, 0.1170203406278252e+01, 0.1168213974642164e+01, 0.3087619670750762e-02, 0.1632777609571200e-02, 0.6177683213554945e+00, 0.6186363605423986e+00, 0.2500816796475141e-02, 0.2057564358223963e-02, 0.5089873548304448e+00, 0.5069297904722210e+00, 0.1826193617023393e-15, 0.4336141378669131e-16, 0.1203661659999391e+00, 0.1203661801217644e+00, 0.1412182528886294e-07, 0.2203879431304952e-08, 0.1878457941299617e-01, 0.1878457720911678e-01, 0.3956520722700361e-16, 0.4283248425464200e-37, 0.1183377192742860e+01},
			i0:       1,
			n0:       8,
			pp:       1,
			tau:      1.8784576682472597e-002,
			sigma:    1.1833771927428600,
			i0Out:    1,
			n0Out:    8,
			ppOut:    1,
			tauOut:   1.8784576682472597e-002,
			sigmaOut: 1.1833771927428600,
			dminOut:  1.1910047947871760e-010,
			dmin1Out: 0.10158160343929173,
			dmin2Out: 0.48640978493259379,
			dnOut:    1.1910047947871760e-010,
			dnm1Out:  0.10158160343929173,
			dnm2Out:  0.48640978493259379,
		},
		{
			z:        []float64{0.1278815744076516e+01, 0.1297600320758988e+01, 0.2322916695987584e-18, 0.2438256017573345e-18, 0.1223346564804853e+01, 0.1218322613210885e+01, 0.2174969171530204e-01, 0.2380852827644004e-01, 0.1082100757788551e+01, 0.1117557974879486e+01, 0.5481081353963683e-02, 0.5077051306839188e-02, 0.1145581094215299e+01, 0.1168213974642164e+01, 0.8817320773368219e-03, 0.1632777609571200e-02, 0.6010276161408131e+00, 0.6186363605423986e+00, 0.1735428857154564e-02, 0.2057564358223963e-02, 0.4864097849325938e+00, 0.5069297904722210e+00, 0.1073014545319338e-16, 0.4336141378669131e-16, 0.1015816056431712e+00, 0.1203661801217644e+00, 0.4075436992240071e-09, 0.2203879431304952e-08, 0.1191004794787176e-09, 0.1878457720911678e-01, 0.2322916695987584e-18, 0.2438256017573345e-18},
			i0:       1,
			n0:       8,
			pp:       0,
			tau:      1.1910047900088848e-010,
			sigma:    1.2021617694253326,
			i0Out:    1,
			n0Out:    8,
			ppOut:    0,
			tauOut:   1.1910047900088848e-010,
			sigmaOut: 1.2021617694253326,
			dminOut:  0.0000000000000000,
			dmin1Out: 0.10158160552407068,
			dmin2Out: 0.48500827096381138,
			dnOut:    0.0000000000000000,
			dnm1Out:  0.10158160552407068,
			dnm2Out:  0.48500827096381138,
		},
		{
			z:        []float64{0.1278815744076516e+01, 0.1278815743957415e+01, 0.2322916695987584e-18, 0.2222159192042978e-18, 0.1223346564804853e+01, 0.1245096256401054e+01, 0.2174969171530204e-01, 0.1890244048666934e-01, 0.1082100757788551e+01, 0.1068679398536745e+01, 0.5481081353963683e-02, 0.5875497537946494e-02, 0.1145581094215299e+01, 0.1140587328635589e+01, 0.8817320773368219e-03, 0.4646249482278370e-03, 0.6010276161408131e+00, 0.6022984199306394e+00, 0.1735428857154564e-02, 0.1401513849681988e-02, 0.4864097849325938e+00, 0.4850082709638114e+00, 0.1073014545319338e-16, 0.2247354260070927e-17, 0.1015816056431712e+00, 0.1015816059316144e+00, 0.4075436992240071e-09, 0.4778291260603437e-18, 0.1191004794787176e-09, 0.0000000000000000e+00, 0.2322916695987584e-18, 0.2222159192042978e-18, 0.1183377192742860e+01},
			i0:       1,
			n0:       8,
			pp:       1,
			tau:      -0.0000000000000000,
			sigma:    1.2021617695444331,
			i0Out:    1,
			n0Out:    8,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.2021617695444331,
			dminOut:  0.0000000000000000,
			dmin1Out: 0.10158160593161437,
			dmin2Out: 0.48388184445926297,
			dnOut:    0.0000000000000000,
			dnm1Out:  0.10158160593161437,
			dnm2Out:  0.48388184445926297,
		},
		{
			z:        []float64{0.1278815743957415e+01, 0.1278815743957415e+01, 0.2163565864913247e-18, 0.2222159192042978e-18, 0.1263998696887723e+01, 0.1245096256401054e+01, 0.1598154237018549e-01, 0.1890244048666934e-01, 0.1058573353704506e+01, 0.1068679398536745e+01, 0.6330707284251229e-02, 0.5875497537946494e-02, 0.1134721246299565e+01, 0.1140587328635589e+01, 0.2466181655543824e-03, 0.4646249482278370e-03, 0.6034533156147670e+00, 0.6022984199306394e+00, 0.1126426504548419e-02, 0.1401513849681988e-02, 0.4838818444592630e+00, 0.4850082709638114e+00, 0.4717884282068346e-18, 0.2247354260070927e-17, 0.1015816059316144e+00, 0.1015816059316144e+00, 0.0000000000000000e+00, 0.4778291260603437e-18},
			i0:       1,
			n0:       7,
			pp:       0,
			tau:      -0.0000000000000000,
			sigma:    1.2021617695444331,
			i0Out:    1,
			n0Out:    7,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.2021617695444331,
			dminOut:  0.10158160593161437,
			dmin1Out: 0.48298010023310722,
			dmin2Out: 0.60332139679048247,
			dnOut:    0.10158160593161437,
			dnm1Out:  0.48298010023310722,
			dnm2Out:  0.60332139679048247,
		},
		{
			z:        []float64{0.1278815743957415e+01, 0.1278815743957415e+01, 0.2163565864913247e-18, 0.2138497627045302e-18, 0.1263998696887723e+01, 0.1279980239257909e+01, 0.1598154237018549e-01, 0.1321710631563048e-01, 0.1058573353704506e+01, 0.1051686954673127e+01, 0.6330707284251229e-02, 0.6830538334266978e-02, 0.1134721246299565e+01, 0.1128137326130852e+01, 0.2466181655543824e-03, 0.1319188242844840e-03, 0.6034533156147670e+00, 0.6044478232950309e+00, 0.1126426504548419e-02, 0.9017442261557238e-03, 0.4838818444592630e+00, 0.4829801002331072e+00, 0.4717884282068346e-18, 0.9922774494036444e-19, 0.1015816059316144e+00, 0.1015816059316144e+00, 0.0000000000000000e+00, 0.2138497627045302e-18, 0.1202161769544433e+01},
			i0:       1,
			n0:       7,
			pp:       1,
			tau:      0.10158160593161437,
			sigma:    1.2021617695444331,
			i0Out:    1,
			n0Out:    7,
			ppOut:    1,
			tauOut:   0.10158160593161437,
			sigmaOut: 1.2021617695444331,
			dminOut:  0.0000000000000000,
			dmin1Out: 0.38053382595775820,
			dmin2Out: 0.50278793042091730,
			dnOut:    0.0000000000000000,
			dnm1Out:  0.38053382595775820,
			dnm2Out:  0.50278793042091730,
		},
		{
			z:        []float64{0.1177234138025801e+01, 0.1278815743957415e+01, 0.2325140442247288e-18, 0.2138497627045302e-18, 0.1191615739641925e+01, 0.1279980239257909e+01, 0.1166505092896250e-01, 0.1321710631563048e-01, 0.9452708361468171e+00, 0.1051686954673127e+01, 0.8151933771558135e-02, 0.6830538334266978e-02, 0.1018535705251965e+01, 0.1128137326130852e+01, 0.7828694249915420e-04, 0.1319188242844840e-03, 0.5036896746470730e+00, 0.6044478232950309e+00, 0.8646683437346259e-03, 0.9017442261557238e-03, 0.3805338259577582e+00, 0.4829801002331072e+00, 0.2648835135390502e-19, 0.9922774494036444e-19, 0.0000000000000000e+00, 0.1015816059316144e+00, 0.2325140442247288e-18, 0.2138497627045302e-18},
			i0:       1,
			n0:       7,
			pp:       0,
			tau:      -0.0000000000000000,
			sigma:    1.3037433754760475,
			i0Out:    1,
			n0Out:    7,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.3037433754760475,
			dminOut:  0.0000000000000000,
			dmin1Out: 0.37988164443454930,
			dmin2Out: 0.50365062581387521,
			dnOut:    0.0000000000000000,
			dnm1Out:  0.37988164443454930,
			dnm2Out:  0.50365062581387521,
		},
		{
			z:        []float64{0.1177234138025801e+01, 0.1177234138025801e+01, 0.2325140442247288e-18, 0.2353545363971710e-18, 0.1191615739641925e+01, 0.1203280790570888e+01, 0.1166505092896250e-01, 0.9163806595868684e-02, 0.9452708361468171e+00, 0.9442589633225065e+00, 0.8151933771558135e-02, 0.8793176380307672e-02, 0.1018535705251965e+01, 0.1009820815814156e+01, 0.7828694249915420e-04, 0.3904883319791864e-04, 0.5036896746470730e+00, 0.5045152941576099e+00, 0.8646683437346259e-03, 0.6521815232088979e-03, 0.3805338259577582e+00, 0.3798816444345493e+00, 0.2648835135390502e-19, 0.0000000000000000e+00, 0.1303743375476047e+01},
			i0:       1,
			n0:       6,
			pp:       1,
			tau:      -0.0000000000000000,
			sigma:    1.3037433754760475,
			i0Out:    1,
			n0Out:    6,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.3037433754760475,
			dminOut:  0.37939119035907048,
			dmin1Out: 0.50449560273072791,
			dmin2Out: 0.93712213699945368,
			dnOut:    0.37939119035907048,
			dnm1Out:  0.50449560273072791,
			dnm2Out:  1.0004335774214996,
		},
		{
			z:        []float64{0.1177234138025801e+01, 0.1177234138025801e+01, 0.2405618249359890e-18, 0.2353545363971710e-18, 0.1212444597166756e+01, 0.1203280790570888e+01, 0.7136826323052847e-02, 0.9163806595868684e-02, 0.9459153133797613e+00, 0.9442589633225065e+00, 0.9387238392656353e-02, 0.8793176380307672e-02, 0.1000472626254697e+01, 0.1009820815814156e+01, 0.1969142688202247e-04, 0.3904883319791864e-04, 0.5051477842539368e+00, 0.5045152941576099e+00, 0.4904540754787929e-03, 0.6521815232088979e-03, 0.3793911903590705e+00, 0.3798816444345493e+00, 0.2405618249359890e-18, 0.2353545363971710e-18},
			i0:       1,
			n0:       6,
			pp:       0,
			tau:      0.37791671367087804,
			sigma:    1.3037433754760475,
			i0Out:    1,
			n0Out:    6,
			ppOut:    0,
			tauOut:   0.37791671367087804,
			sigmaOut: 1.3037433754760475,
			dminOut:  1.7417124602014944e-005,
			dmin1Out: 0.12721465844156904,
			dmin2Out: 0.55997778901175921,
			dnOut:    1.7417124602014944e-005,
			dnm1Out:  0.12721465844156904,
			dnm2Out:  0.60606091447901145,
		},
		{
			z:        []float64{0.1177234138025801e+01, 0.7993174243549228e+00, 0.2405618249359890e-18, 0.3648961927279404e-18, 0.1212444597166756e+01, 0.8416647098189309e+00, 0.7136826323052847e-02, 0.8020810697124018e-02, 0.9459153133797613e+00, 0.5693650274044155e+00, 0.9387238392656353e-02, 0.1649499810480807e-01, 0.1000472626254697e+01, 0.6060806059058935e+00, 0.1969142688202247e-04, 0.1641214148963635e-04, 0.5051477842539368e+00, 0.1277051125170478e+00, 0.4904540754787929e-03, 0.1457059563590431e-02, 0.3793911903590705e+00, 0.1741712460201494e-04, 0.2405618249359890e-18, 0.3648961927279404e-18, 0.1303743375476047e+01},
			i0:       1,
			n0:       6,
			pp:       1,
			tau:      1.7220607103034587e-005,
			sigma:    1.6816600891469256,
			i0Out:    1,
			n0Out:    6,
			ppOut:    1,
			tauOut:   1.7220607103034587e-005,
			sigmaOut: 1.6816600891469256,
			dminOut:  5.8496704963934779e-012,
			dmin1Out: 0.12768433261716639,
			dmin2Out: 0.56397303984076830,
			dnOut:    5.8496704963934779e-012,
			dnm1Out:  0.12768433261716639,
			dnm2Out:  0.58884056200405466,
		},
		{
			z:        []float64{0.7993002037478197e+00, 0.7993174243549228e+00, 0.3842364192156412e-18, 0.3648961927279404e-18, 0.8496682999089519e+00, 0.8416647098189309e+00, 0.5374766956544110e-02, 0.8020810697124018e-02, 0.5804680379455763e+00, 0.5693650274044155e+00, 0.1722282329473577e-01, 0.1649499810480807e-01, 0.5888569741455443e+00, 0.6060806059058935e+00, 0.3559292778388127e-05, 0.1641214148963635e-04, 0.1291413921807568e+00, 0.1277051125170478e+00, 0.1965116493098606e-06, 0.1457059563590431e-02, 0.5849670496393478e-11, 0.1741712460201494e-04, 0.3842364192156412e-18, 0.3648961927279404e-18},
			i0:       1,
			n0:       6,
			pp:       0,
			tau:      5.8496615949827599e-012,
			sigma:    1.6816773097540285,
			i0Out:    1,
			n0Out:    6,
			ppOut:    0,
			tauOut:   5.8496615949827599e-012,
			sigmaOut: 1.6816773097540285,
			dminOut:  5.2413685591311719e-023,
			dmin1Out: 0.12914058828956335,
			dmin2Out: 0.57178448001987381,
			dnOut:    5.2413685591311719e-023,
			dnm1Out:  0.12914058828956335,
			dnm2Out:  0.57178448001987381,
		},
		{
			z:        []float64{0.7993002037478197e+00, 0.7993002037419701e+00, 0.3842364192156412e-18, 0.4084491703488284e-18, 0.8496682999089519e+00, 0.8550430668596465e+00, 0.5374766956544110e-02, 0.3648799166500929e-02, 0.5804680379455763e+00, 0.5940420620679615e+00, 0.1722282329473577e-01, 0.1707249411982080e-01, 0.5888569741455443e+00, 0.5717880393126522e+00, 0.3559292778388127e-05, 0.8038853438286423e-06, 0.1291413921807568e+00, 0.1291407848012127e+00, 0.1965116493098606e-06, 0.8901358303923820e-17, 0.5849670496393478e-11, 0.5241368559131172e-22, 0.3842364192156412e-18, 0.4084491703488284e-18, 0.1303743375476047e+01},
			i0:       1,
			n0:       6,
			pp:       1,
			tau:      5.2413685591311714e-023,
			sigma:    1.6816773097598783,
			i0Out:    1,
			n0Out:    6,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.6816773097598783,
			dminOut:  5.2413685591311719e-023,
			dmin1Out: 0.12914059800027977,
			dmin2Out: 0.55574794205624078,
			dnOut:    5.2413685591311719e-023,
			dnm1Out:  0.12914059800027977,
			dnm2Out:  0.55574794205624078,
		},
		{
			z:        []float64{0.7993002037419701e+00, 0.7993002037419701e+00, 0.4369342452764876e-18, 0.4084491703488284e-18, 0.8586918660261474e+00, 0.8550430668596465e+00, 0.2524235138002424e-02, 0.3648799166500929e-02, 0.6085903210497798e+00, 0.5940420620679615e+00, 0.1604009725641142e-01, 0.1707249411982080e-01, 0.5557487459415846e+00, 0.5717880393126522e+00, 0.1868009328861996e-06, 0.8038853438286423e-06, 0.1291405980002798e+00, 0.1291407848012127e+00, 0.3612752323451872e-38, 0.8901358303923820e-17},
			i0:       1,
			n0:       5,
			pp:       0,
			tau:      0.12914051019182277,
			sigma:    1.6816773097598783,
			i0Out:    1,
			n0Out:    5,
			ppOut:    0,
			tauOut:   0.12914051019182277,
			sigmaOut: 1.6816773097598783,
			dminOut:  2.8760335918365243e-008,
			dmin1Out: 0.40854091047624980,
			dmin2Out: 0.47735135942481804,
			dnOut:    2.8760335918365243e-008,
			dnm1Out:  0.40854091047624980,
			dnm2Out:  0.47735135942481804,
		},
		{
			z:        []float64{0.7993002037419701e+00, 0.6701596935501473e+00, 0.4369342452764876e-18, 0.5598544436768911e-18, 0.8586918660261474e+00, 0.7320755909723271e+00, 0.2524235138002424e-02, 0.2098451433139096e-02, 0.6085903210497798e+00, 0.4933914566812295e+00, 0.1604009725641142e-01, 0.1806732527351204e-01, 0.5557487459415846e+00, 0.4085410972771827e+00, 0.1868009328861996e-06, 0.5904812108429529e-07, 0.1291405980002798e+00, 0.2876033591836524e-07, 0.3612752323451872e-38, 0.5598544436768911e-18, 0.1681677309759878e+01},
			i0:       1,
			n0:       5,
			pp:       1,
			tau:      2.8760330654564486e-008,
			sigma:    1.8108178199517009,
			i0Out:    1,
			n0Out:    5,
			ppOut:    1,
			tauOut:   2.8760330654564486e-008,
			sigmaOut: 1.8108178199517009,
			dminOut:  9.5429721930147733e-016,
			dmin1Out: 0.39406941605409296,
			dmin2Out: 0.49198119275848701,
			dnOut:    9.5429721930147733e-016,
			dnm1Out:  0.39406941605409296,
			dnm2Out:  0.49198119275848701,
		},
		{
			z:        []float64{0.6701596647898166e+00, 0.6701596935501473e+00, 0.6115792910959321e-18, 0.5598544436768911e-18, 0.7341740136451356e+00, 0.7320755909723271e+00, 0.1410235162411720e-02, 0.2098451433139096e-02, 0.5100485180319990e+00, 0.4933914566812295e+00, 0.1447165246275905e-01, 0.1806732527351204e-01, 0.3940694751022140e+00, 0.4085410972771827e+00, 0.4309503539425748e-14, 0.5904812108429529e-07, 0.9542972193014773e-15, 0.2876033591836524e-07, 0.6115792910959321e-18, 0.5598544436768911e-18},
			i0:       1,
			n0:       5,
			pp:       0,
			tau:      9.5429721930146451e-016,
			sigma:    1.8108178487120317,
			i0Out:    1,
			n0Out:    5,
			ppOut:    0,
			tauOut:   9.5429721930146451e-016,
			sigmaOut: 1.8108178487120317,
			dminOut:  1.9721522630525295e-030,
			dmin1Out: 0.38317668529976806,
			dmin2Out: 0.50907067170667131,
			dnOut:    1.9721522630525295e-030,
			dnm1Out:  0.38317668529976806,
			dnm2Out:  0.50907067170667131,
		},
		{
			z:        []float64{0.6701596647898166e+00, 0.6701596647898156e+00, 0.6115792910959321e-18, 0.6699979816704878e-18, 0.7341740136451356e+00, 0.7355842488075462e+00, 0.1410235162411720e-02, 0.9778463253267723e-03, 0.5100485180319990e+00, 0.5235423241694304e+00, 0.1447165246275905e-01, 0.1089278980244500e-01, 0.3940694751022140e+00, 0.3831766852997724e+00, 0.4309503539425748e-14, 0.1073277003016631e-28, 0.1810817848712033e+01},
			i0:       1,
			n0:       4,
			pp:       1,
			tau:      0.31895966905528556,
			sigma:    1.8108178487120326,
			i0Out:    1,
			n0Out:    4,
			ppOut:    1,
			tauOut:   0.31895966905528556,
			sigmaOut: 1.8108178487120326,
			dminOut:  4.4735699115826311e-002,
			dmin1Out: 0.20335674284478694,
			dmin2Out: 0.35119999573453003,
			dnOut:    4.4735699115826311e-002,
			dnm1Out:  0.20335674284478694,
			dnm2Out:  0.41662457975226058,
		},
		{
			z:        []float64{0.3511999957345300e+00, 0.6701596647898156e+00, 0.1403302870260262e-17, 0.6699979816704878e-18, 0.4176024260775874e+00, 0.7355842488075462e+00, 0.1225912269357841e-02, 0.9778463253267723e-03, 0.2142495326472319e+00, 0.5235423241694304e+00, 0.1948131712866055e-01, 0.1089278980244500e-01, 0.4473569911582631e-01, 0.3831766852997724e+00, 0.1403302870260262e-17, 0.6699979816704878e-18},
			i0:       1,
			n0:       4,
			pp:       0,
			tau:      3.9897804510299034e-002,
			sigma:    2.1297775177673182,
			i0Out:    1,
			n0Out:    4,
			ppOut:    0,
			tauOut:   3.9897804510299034e-002,
			sigmaOut: 2.1297775177673182,
			dminOut:  3.2556800493375310e-004,
			dmin1Out: 0.17365859019389857,
			dmin2Out: 0.31130219122423097,
			dnOut:    3.2556800493375310e-004,
			dnm1Out:  0.17365859019389857,
			dnm2Out:  0.37770462156728835,
		},
		{
			z:        []float64{0.3511999957345300e+00, 0.3113021912242310e+00, 0.1403302870260262e-17, 0.1882488140663987e-17, 0.4176024260775874e+00, 0.3789305338366462e+00, 0.1225912269357841e-02, 0.6931379430343076e-03, 0.2142495326472319e+00, 0.1931399073225591e+00, 0.1948131712866055e-01, 0.4512326600593524e-02, 0.4473569911582631e-01, 0.3255680049337531e-03, 0.1403302870260262e-17, 0.1882488140663987e-17, 0.1810817848712033e+01},
			i0:       1,
			n0:       4,
			pp:       1,
			tau:      3.1797943619548145e-004,
			sigma:    2.1696753222776173,
			i0Out:    1,
			n0Out:    4,
			ppOut:    1,
			tauOut:   3.1797943619548145e-004,
			sigmaOut: 2.1696753222776173,
			dminOut:  1.3065740500455263e-007,
			dmin1Out: 0.19246898670783216,
			dmin2Out: 0.31098421178803548,
			dnOut:    1.3065740500455263e-007,
			dnm1Out:  0.19246898670783216,
			dnm2Out:  0.37861255440045072,
		},
		{
			z:        []float64{0.3109842117880355e+00, 0.3113021912242310e+00, 0.2293789231233262e-17, 0.1882488140663987e-17, 0.3793056923434850e+00, 0.3789305338366462e+00, 0.3529411785314453e-03, 0.6931379430343076e-03, 0.1969813133084257e+00, 0.1931399073225591e+00, 0.7457911333267062e-05, 0.4512326600593524e-02, 0.1306574050045526e-06, 0.3255680049337531e-03, 0.2293789231233262e-17, 0.1882488140663987e-17},
			i0:       1,
			n0:       4,
			pp:       0,
			tau:      1.3065240973281283e-007,
			sigma:    2.1699933017138124,
			i0Out:    1,
			n0Out:    4,
			ppOut:    0,
			tauOut:   1.3065240973281283e-007,
			sigmaOut: 2.1699933017138124,
			dminOut:  4.4031878668096113e-014,
			dmin1Out: 0.19679806330970273,
			dmin2Out: 0.31098408113562576,
			dnOut:    4.4031878668096113e-014,
			dnm1Out:  0.19679806330970273,
			dnm2Out:  0.37930556169107532,
		},
		{
			z:        []float64{0.3109842117880355e+00, 0.3109840811356258e+00, 0.2293789231233262e-17, 0.2797722987188916e-17, 0.3793056923434850e+00, 0.3796585028696068e+00, 0.3529411785314453e-03, 0.1831193463132450e-03, 0.1969813133084257e+00, 0.1968055212210360e+00, 0.7457911333267062e-05, 0.4951239861123181e-11, 0.1306574050045526e-06, 0.4403187866809611e-13, 0.2293789231233262e-17, 0.2797722987188916e-17, 0.1810817848712033e+01},
			i0:       1,
			n0:       4,
			pp:       1,
			tau:      4.4031878666982759e-014,
			sigma:    2.1699934323662222,
			i0Out:    1,
			n0Out:    4,
			ppOut:    1,
			tauOut:   4.4031878666982759e-014,
			sigmaOut: 2.1699934323662222,
			dminOut:  5.0676424551397798e-027,
			dmin1Out: 0.19671064247077341,
			dmin2Out: 0.31098408113558174,
			dnOut:    5.0676424551397798e-027,
			dnm1Out:  0.19671064247077341,
			dnm2Out:  0.37965850286956282,
		},
		{
			z:        []float64{0.3109840811355817e+00, 0.3109840811356258e+00, 0.3415542419024794e-17, 0.2797722987188916e-17, 0.3798416222158761e+00, 0.3796585028696068e+00, 0.9487875021856221e-04, 0.1831193463132450e-03, 0.1967106424757246e+00, 0.1968055212210360e+00, 0.1108289770587888e-23, 0.4951239861123181e-11, 0.5067642455139780e-26, 0.4403187866809611e-13, 0.3415542419024794e-17, 0.2797722987188916e-17},
			i0:       1,
			n0:       4,
			pp:       0,
			tau:      5.0676424551397798e-027,
			sigma:    2.1699934323662662,
			i0Out:    1,
			n0Out:    4,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 2.1699934323662662,
			dminOut:  5.0676424551397798e-027,
			dmin1Out: 0.19666151937261311,
			dmin2Out: 0.31098408113558174,
			dnOut:    5.0676424551397798e-027,
			dnm1Out:  0.19666151937261311,
			dnm2Out:  0.37984162221587608,
		},
		{
			z:        []float64{0.3109840811355817e+00, 0.3109840811355817e+00, 0.3415542419024794e-17, 0.4171805735046273e-17, 0.3798416222158761e+00, 0.3799365009660947e+00, 0.9487875021856221e-04, 0.4912310311151891e-04, 0.1967106424757246e+00, 0.1966615193726131e+00, 0.1108289770587888e-23, 0.2855879641297252e-49, 0.2169993432366266e+01},
			i0:       1,
			n0:       3,
			pp:       1,
			tau:      9.8330759686306557e-002,
			sigma:    2.1699934323662662,
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			tauOut:   9.8330759686306557e-002,
			sigmaOut: 2.1699934323662662,
			dminOut:  9.8296460175394978e-002,
			dmin1Out: 0.21265332144927518,
			dmin2Out: 0.21265332144927518,
			dnOut:    9.8296460175394978e-002,
			dnm1Out:  0.28160574127978810,
			dnm2Out:  0.21265332144927518,
		},
		{
			z:        []float64{0.2126533214492752e+00, 0.3109840811355817e+00, 0.7453545812882342e-17, 0.4171805735046273e-17, 0.2816548643828996e+00, 0.3799365009660947e+00, 0.3429951091160839e-04, 0.4912310311151891e-04, 0.9829646017539498e-01, 0.1966615193726131e+00, 0.3799365009660947e+00, 0.2855879641297252e-49},
			i0:       1,
			n0:       3,
			pp:       0,
			tau:      9.7172513485343004e-002,
			sigma:    2.2683241920525727,
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			tauOut:   9.7172513485343004e-002,
			sigmaOut: 2.2683241920525727,
			dminOut:  1.1056745151512559e-003,
			dmin1Out: 0.11548080796393217,
			dmin2Out: 0.11548080796393217,
			dnOut:    1.1056745151512559e-003,
			dnm1Out:  0.18448235089755655,
			dnm2Out:  0.11548080796393217,
		},
		{
			z:        []float64{0.2126533214492752e+00, 0.1154808079639322e+00, 0.7453545812882342e-17, 0.1817901582187390e-16, 0.2816548643828996e+00, 0.1845166504084682e+00, 0.3429951091160839e-04, 0.1827217490071947e-04, 0.9829646017539498e-01, 0.1105674515151256e-02, 0.3799365009660947e+00, 0.2816548643828996e+00, 0.2169993432366266e+01},
			i0:       1,
			n0:       3,
			pp:       1,
			tau:      1.0942861809085330e-003,
			sigma:    2.3654967055379159,
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			tauOut:   1.0942861809085330e-003,
			sigmaOut: 2.3654967055379159,
			dminOut:  1.1278200095129901e-005,
			dmin1Out: 0.11438652178302365,
			dmin2Out: 0.11438652178302365,
			dnOut:    1.1278200095129901e-005,
			dnm1Out:  0.18342236422755959,
			dnm2Out:  0.11438652178302365,
		},
		{
			z:        []float64{0.1143865217830237e+00, 0.1154808079639322e+00, 0.2932453102768040e-16, 0.1817901582187390e-16, 0.1834406364024603e+00, 0.1845166504084682e+00, 0.1101341475930520e-06, 0.1827217490071947e-04, 0.1127820009512990e-04, 0.1105674515151256e-02, 0.1845166504084682e+00, 0.2816548643828996e+00},
			i0:       1,
			n0:       3,
			pp:       0,
			tau:      1.1269238360546607e-005,
			sigma:    2.3665909917188244,
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			tauOut:   1.1269238360546607e-005,
			sigmaOut: 2.3665909917188244,
			dminOut:  8.9549629620034135e-009,
			dmin1Out: 0.11437525254466312,
			dmin2Out: 0.11437525254466312,
			dnOut:    8.9549629620034135e-009,
			dnm1Out:  0.18342936716409974,
			dnm2Out:  0.11437525254466312,
		},
		{
			z:        []float64{0.1143865217830237e+00, 0.1143752525446631e+00, 0.2932453102768040e-16, 0.4703212027287794e-16, 0.1834406364024603e+00, 0.1834294772982473e+00, 0.1101341475930520e-06, 0.6771621290952006e-11, 0.1127820009512990e-04, 0.8954962962003413e-08, 0.1845166504084682e+00, 0.1834406364024603e+00, 0.2169993432366266e+01},
			i0:       1,
			n0:       3,
			pp:       1,
			tau:      8.9549072084035346e-009,
			sigma:    2.3666022609571851,
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			tauOut:   8.9549072084035346e-009,
			sigmaOut: 2.3666022609571851,
			dminOut:  5.5753269291151117e-014,
			dmin1Out: 0.11437524358975594,
			dmin2Out: 0.11437524358975594,
			dnOut:    5.5753269291151117e-014,
			dnm1Out:  0.18342946834334006,
			dnm2Out:  0.11437524358975594,
		},
		{
			z:        []float64{0.1143752435897560e+00, 0.1143752525446631e+00, 0.7542783706608855e-16, 0.4703212027287794e-16, 0.1834294683501117e+00, 0.1834294772982473e+00, 0.3305882004599510e-18, 0.6771621290952006e-11, 0.5575326929115112e-13, 0.8954962962003413e-08, 0.1834294772982473e+00, 0.1834406364024603e+00},
			i0:       1,
			n0:       3,
			pp:       0,
			tau:      5.5753269214454873e-014,
			sigma:    2.3666022699120921,
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			tauOut:   5.5753269214454873e-014,
			sigmaOut: 2.3666022699120921,
			dminOut:  7.6696244203643861e-023,
			dmin1Out: 0.11437524358970023,
			dmin2Out: 0.11437524358970023,
			dnOut:    7.6696244203643861e-023,
			dnm1Out:  0.18342946835005580,
			dnm2Out:  0.11437524358970023,
		},
		{
			z:        []float64{0.1648283185136998e+01, 0.1396221235720571e+01, 0.1712714336271993e+00, 0.2520619494164272e+00, 0.1510753432847732e+01, 0.1119977945086946e+01, 0.3792800633372563e+00, 0.5620469213879850e+00, 0.1328441621586708e+01, 0.1019485448443405e+01, 0.1388998759717073e+00, 0.6882362364805590e+00, 0.4466938077796418e+00, 0.2681061628164644e+00, 0.7093206047768255e+00, 0.3174875209348847e+00, 0.6843112870203156e+00, 0.9979892152967577e+00, 0.6490582981441884e+00, 0.3956426765003833e+00, 0.5085572738629487e+00, 0.1122623886995757e+01, 0.9750235054014829e-02, 0.3499168501137979e-01, 0.1364886053450573e+00, 0.1417066070690837e+00, 0.2301225778544498e-01, 0.4532233329988395e-02, 0.1646009972289452e+01, 0.6930161671496210e+00, 0.2362515608142310e+00, 0.9760060629252760e+00, 0.5818602562677768e+00, 0.3984323866837953e+00, 0.1797665269485310e-01, 0.4196794303982125e+00, 0.5600419521166516e+00, 0.2492354636952108e-01, 0.2195137569256029e+00, 0.5530950584419837e+00, 0.4184071984843414e+00, 0.2222708575473020e+00, 0.2727864547293006e+00, 0.4156500978626423e+00, 0.6774373914466536e-01, 0.2745959086613283e+00, 0.1050967099374242e+00, 0.6593428521263771e-01, 0.2040338718098096e+00, 0.1079809097801335e+00, 0.1271971985482246e+00, 0.2011496719671002e+00, 0.4444741998443960e-01, 0.1290210252363728e+00, 0.5776327498150620e+00, 0.4262359329629137e-01, 0.3402556968467140e+00, 0.6023491555328507e+00, 0.1086565805630269e+00, 0.3155392911289253e+00, 0.9382999256694983e+00, 0.1171677238805356e+00, 0.7901625299559836e+00, 0.9297887823519896e+00, 0.6095853796269167e+00, 0.7973955560628040e+00, 0.8018125008387630e+00, 0.6023523535200964e+00, 0.2443177602187348e-01, 0.8114406374558937e+00, 0.2277830378453201e+00, 0.1480363940474286e-01, 0.6969081780841352e+00, 0.3759308106650992e+00, 0.8012406122589412e-01, 0.5487604052643561e+00, 0.9781022865072954e-01, 0.1017549972519246e+00, 0.2961248981181939e-03, 0.7617929262469909e-01, 0.8408398800007548e-04, 0.3802088861182694e-03, 0.0000000000000000e+00, 0.0000000000000000e+00},
			i0:       1,
			n0:       21,
			pp:       0,
			tau:      0.0000000000000000,
			sigma:    0.0000000000000000,
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			dminOut:  7.8766827084626452e-005,
			dmin1Out: 1.2363512593342330e-003,
			dmin2Out: 1.2363512593342330e-003,
			dnOut:    7.8766827084626452e-005,
			dnm1Out:  4.3867054272813191e-003,
			dnm2Out:  3.7622286348031123e-003,
		},
	} {
		// TODO(btracey): Check other outputs
		i0 := test.i0 - 1
		n0 := test.n0 - 1
		z := make([]float64, len(test.z))
		copy(z, test.z)

		i0Out, n0Out, ppOut, tauOut, sigmaOut, dminOut, dmin1Out, dmin2Out, dnOut, dnm1Out, dnm2Out := impl.Dlasq5(i0, n0, z, test.pp, test.tau, test.sigma)

		if i0Out != test.i0Out-1 {
			t.Errorf("Wrong i0. Want %v, got %v", test.n0Out, n0Out)
		}
		if n0Out != test.n0Out-1 {
			t.Errorf("Wrong n0. Want %v, got %v", test.n0Out, n0Out)
		}
		if ppOut != test.ppOut {
			t.Errorf("Wrong pp. Want %v, got %v", test.ppOut, ppOut)
		}
		if !floats.EqualWithinAbsOrRel(tauOut, test.tauOut, dTol, dTol) {
			t.Errorf("Wrong tau. Want %v, got %v", test.tauOut, tauOut)
		}
		if !floats.EqualWithinAbsOrRel(sigmaOut, test.sigmaOut, dTol, dTol) {
			t.Errorf("Wrong tau. Want %v, got %v", test.sigmaOut, sigmaOut)
		}

		if !floats.EqualWithinAbsOrRel(dminOut, test.dminOut, dTol, dTol) {
			t.Errorf("Wrong dmin. Want %v, got %v", test.dminOut, dminOut)
		}
		if !floats.EqualWithinAbsOrRel(dmin1Out, test.dmin1Out, dTol, dTol) {
			t.Errorf("Wrong dmin1. Want %v, got %v", test.dmin1Out, dmin1Out)
		}
		if !floats.EqualWithinAbsOrRel(dmin2Out, test.dmin2Out, dTol, dTol) {
			t.Errorf("Wrong dmin2. Want %v, got %v", test.dmin2Out, dmin2Out)
		}
		if !floats.EqualWithinAbsOrRel(dnOut, test.dnOut, dTol, dTol) {
			t.Errorf("Wrong dn. Want %v, got %v", test.dnOut, dnOut)
		}
		if !floats.EqualWithinAbsOrRel(dnm1Out, test.dnm1Out, dTol, dTol) {
			t.Errorf("Wrong dnm1. Want %v, got %v", test.dnm1Out, dnm1Out)
		}
		if !floats.EqualWithinAbsOrRel(dnm2Out, test.dnm2Out, dTol, dTol) {
			t.Errorf("Wrong dnm2. Want %v, got %v", test.dnm2Out, dnm2Out)
		}
	}
}
